//==============================================================================
//	
//	Copyright (c) 2002-
//	Authors:
//	* Andrew Hinton <ug60axh@cs.bham.ac.uk> (University of Birmingham)
//	* Dave Parker <david.parker@comlab.ox.ac.uk> (University of Oxford, formerly University of Birmingham)
//	
//------------------------------------------------------------------------------
//	
//	This file is part of PRISM.
//	
//	PRISM is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//	
//	PRISM is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	
//	You should have received a copy of the GNU General Public License
//	along with PRISM; if not, write to the Free Software Foundation,
//	Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//	
//==============================================================================

package userinterface.model;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.File;
import java.util.HashMap;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

import io.ModelExportFormat;
import io.ModelExportOptions;
import io.ModelExportTask;
import io.ModelExportTask.ModelExportEntity;
import prism.ModelType;
import prism.PrismSettings;
import prism.PrismSettingsListener;
import userinterface.GUIClipboardEvent;
import userinterface.GUIPlugin;
import userinterface.GUIPrism;
import userinterface.simulator.GUISimulator;
import userinterface.util.GUIComputationEvent;
import userinterface.util.GUIEvent;
import userinterface.util.GUIExitEvent;
import userinterface.util.GUIUndoManager;

@SuppressWarnings("serial")
public class GUIMultiModel extends GUIPlugin implements PrismSettingsListener
{
	//Constants
	public static final int CONTINUE = 0;
	public static final int CANCEL = 1;

	//GUI
	private JTextField fileTextField;
	private JMenu modelMenu, newMenu, viewMenu, exportMenu, computeMenu, computeExportMenu;
	private JMenu exportSSMenu, exportTrMenu;
	private AbstractAction viewStates, viewTrans, viewObs, viewStateRewards, viewTransRewards, viewLabels, viewPrismCode;
	private AbstractAction computeSS, computeTr, newPRISMModel;
	private AbstractAction newPEPAModel, loadModel, reloadModel, saveModel, saveAsModel, parseModel, buildModel;
	private AbstractAction exportStatesPlain, exportStatesMatlab,
			exportTransPlain, exportTransMatlab, exportTransDot, exportTransUMB, exportObsPlain, exportObsMatlab, exportStateRewardsPlain, exportStateRewardsMatlab,
			exportTransRewardsPlain, exportTransRewardsMatlab, exportLabelsPlain, exportLabelsMatlab,
			exportSSPlain, exportSSMatlab, exportTrPlain, exportTrMatlab;
	private JPopupMenu popup;
	//Contents
	private GUIMultiModelHandler handler;
	private Map<String,FileFilter> modelFilters;
	private Map<String,FileFilter> staFilters;
	private Map<String,FileFilter> traFilters;
	private Map<String,FileFilter> obsFilters;
	private Map<String,FileFilter> labFilters;
	private FileFilter textFilter;
	private FileFilter matlabFilter;
	private FileFilter dotFilter;
	private FileFilter umbFilter;
	//State
	private boolean computing = false;
	private boolean initialised = false;

    // simulator
    private GUISimulator sim;

	/** Creates a new instance of GUIMultiModel */
	public GUIMultiModel(GUIPrism pr)
	{
		super(pr);
		initialised = false;
		initComponents();
		initialised = true;
		doEnables();
	}

	public void takeCLArgs(String args[])
	{
		if (args.length > 0) {
			File f = new File(args[0]);
			// load the specified model file
			handler.loadModel(f, false);
			// set the default directory of the file chooser
			File dir = f.getParentFile();
			if (dir == null)
				dir = new File(".");
			getGUI().getChooser().setCurrentDirectory(dir);
		}
	}

	@Override
	public void onInitComponentsCompleted()
	{
		// forward to multi-model handler
		handler.onInitComponentsCompleted();
	}

	public GUIMultiModelHandler getHandler()
	{
		return handler;
	}

	public JPopupMenu getPopup()
	{
		return popup;
	}

	public boolean getComputing()
	{
		return computing;
	}
	
	public void doEnables()
	{
		// do nothing if not initialised yet
		if (!initialised)
			return;
		// setup file label
		int mode = handler.getModelMode();
		String s = "";
		switch (mode) {
		case GUIMultiModelHandler.PRISM_MODE:
			s += "PRISM Model File: ";
			break;
		case GUIMultiModelHandler.PEPA_MODE:
			s += "PEPA Model File: ";
			break;
		}
		s += handler.getActiveFileName();
		if (handler.modified())
			s += "*";
		fileTextField.setText(s);
		// model menu
		newPRISMModel.setEnabled(!computing);
		newPEPAModel.setEnabled(!computing);
		loadModel.setEnabled(!computing);
		reloadModel.setEnabled(!computing && handler.hasActiveFile());
		saveModel.setEnabled(!computing && handler.modified());
		saveAsModel.setEnabled(!computing);
		parseModel.setEnabled(!computing);
		buildModel.setEnabled(!computing);
		viewStates.setEnabled(!computing);
		viewTrans.setEnabled(!computing);
		viewObs.setEnabled(!computing);
		viewStateRewards.setEnabled(!computing);
		viewTransRewards.setEnabled(!computing);
		viewLabels.setEnabled(!computing);
		viewPrismCode.setEnabled(!computing && handler.getParseState() == GUIMultiModelTree.TREE_SYNCHRONIZED_GOOD);
		computeSS.setEnabled(!computing && (handler.getParsedModelType() == ModelType.CTMC || handler.getParsedModelType() == ModelType.DTMC));
		computeTr.setEnabled(!computing && (handler.getParsedModelType() == ModelType.CTMC || handler.getParsedModelType() == ModelType.DTMC));
		exportStatesPlain.setEnabled(!computing);
		exportStatesMatlab.setEnabled(!computing);
		exportTransPlain.setEnabled(!computing);
		exportTransMatlab.setEnabled(!computing);
		exportTransDot.setEnabled(!computing);
		exportTransUMB.setEnabled(!computing);
		exportObsPlain.setEnabled(!computing);
		exportObsMatlab.setEnabled(!computing);
		exportStateRewardsPlain.setEnabled(!computing);
		exportStateRewardsMatlab.setEnabled(!computing);
		exportTransRewardsPlain.setEnabled(!computing);
		exportTransRewardsMatlab.setEnabled(!computing);
		exportLabelsPlain.setEnabled(!computing);
		exportLabelsMatlab.setEnabled(!computing);
	}

	public int doModificationCheck()
	{
		if (!handler.modified())
			return CONTINUE;
		if (!handler.hasActiveFile()) {
			String[] selection = { "Yes", "No", "Cancel" };
			int selectionNo = -1;
			selectionNo = optionPane("Model has been modified.\nDo you wish to save it?", "Question", JOptionPane.OK_CANCEL_OPTION,
					JOptionPane.QUESTION_MESSAGE, selection, selection[0]);
			switch (selectionNo) {
			case 0:
				return a_saveModelAs();
			case 1:
				return CONTINUE;
			case 2:
				return CANCEL;
			default:
				return CANCEL;
			}
		} else {
			String[] selection = { "Yes", "No", "Save As...", "Cancel" };
			int selectionNo = -1;
			selectionNo = optionPane("Model has been modified.\nDo you wish to save it?", "Question", JOptionPane.OK_CANCEL_OPTION,
					JOptionPane.QUESTION_MESSAGE, selection, selection[0]);
			switch (selectionNo) {
			case 0:
				return a_saveModel();
			case 1:
				return CONTINUE;
			case 2:
				return a_saveModelAs();
			case 3:
				return CANCEL;
			default:
				return CANCEL;
			}
		}
	}

	public void showModel(String modelString)
	{
		JDialog diag = new JDialog(getGUI(), false);

		diag.setTitle("Parsed PRISM Model");
		GUITextModelEditor edit = new GUITextModelEditor(modelString, handler);
		edit.setEditable(false);
		edit.setBackground(new Color(224, 255, 255));
		/*
		PrismEditorKit kit = new PrismEditorKit();
		edit.setEditorKitForContentType("text/prism", kit);
		edit.setContentType("text/prism");
		edit.setPreferredSize(new Dimension(1000,300));
		edit.setText(modelString);
		edit.setFont(new Font("monospaced", Font.PLAIN, 12));*/

		JScrollPane scro = new JScrollPane();
		scro.setPreferredSize(new Dimension(640, 300));
		scro.setViewportView(edit);
		diag.getContentPane().add(scro);

		diag.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		diag.pack();
		diag.setLocationRelativeTo(getGUI()); // centre
		diag.setVisible(true);
	}

	//Action methods

	protected void a_newPRISMModel()
	{
		int cont = doModificationCheck();
		if (cont == CONTINUE)
			handler.newPRISMModel();
	}

	protected void a_newPEPAModel()
	{
		int cont = doModificationCheck();
		if (cont == CONTINUE)
			handler.newPEPAModel();
	}

	protected void a_openModel()
	{
		int cont = doModificationCheck();
		if (cont == CONTINUE) {
			String filterName;
			switch (handler.getModelMode()) {
			case GUIMultiModelHandler.PEPA_MODE:
				filterName = "pepa";
				break;
			default:
				filterName = "prism";
			}
			if (showOpenFileDialog(modelFilters.values(), modelFilters.get(filterName)) == JFileChooser.APPROVE_OPTION) {
				File file = getChooserFile();
				if (file == null) {
					error("No file selected");
					return;
				}
				getPrism().getMainLog().resetNumberOfWarnings();
				handler.loadModel(file);
			}
		}
	}

	protected void a_reloadModel()
	{
		int cont = doModificationCheck();
		if (cont == CONTINUE) {
			getPrism().getMainLog().resetNumberOfWarnings();
			handler.reloadActiveFile();
		}
	}

	protected int a_saveModel()
	{
		if (!handler.hasActiveFile()) {
			return a_saveModelAs();
		} else {
			getPrism().getMainLog().resetNumberOfWarnings();
			return handler.saveToActiveFile();
		}
	}

	protected int a_saveModelAs()
	{
		String filterName;
		switch (handler.getModelMode()) {
		case GUIMultiModelHandler.PEPA_MODE:
			filterName = "pepa";
			break;
		default:
			filterName = "prism";
		}
		if (showSaveFileDialog(modelFilters.values(), modelFilters.get(filterName)) != JFileChooser.APPROVE_OPTION) {
			return CANCEL;
		}
		// do save
		return handler.saveToFile(getChooserFile());
	}

	protected void a_refreshParseTree()
	{
		handler.requestParse(true);
	}

	protected void a_build()
	{
		// Reset warnings counter 
		getPrism().getMainLog().resetNumberOfWarnings();
		// Request build
		handler.forceBuild();
	}

	protected void a_exportBuildAs(ModelExportEntity exportEntity, ModelExportFormat exportFormat)
	{
		int res = JFileChooser.CANCEL_OPTION;

		// pop up dialog to select file
		switch (exportFormat) {
			case DOT:
				res = showSaveFileDialog(dotFilter);
				break;
			case UMB:
				res = showSaveFileDialog(umbFilter);
				break;
			case MATLAB:
				res = showSaveFileDialog(matlabFilter);
				break;
			default:
				switch (exportEntity) {
					case STATES:
						res = showSaveFileDialog(staFilters.values(), staFilters.get("sta"));
						break;
					case MODEL:
						res = showSaveFileDialog(traFilters.values(), traFilters.get("tra"));
						break;
					case OBSERVATIONS:
						res = showSaveFileDialog(obsFilters.values(), obsFilters.get("obs"));
						break;
					case LABELS:
						res = showSaveFileDialog(labFilters.values(), labFilters.get("lab"));
						break;
					default:
						res = showSaveFileDialog(textFilter);
				}
				break;
		}
		if (res != JFileChooser.APPROVE_OPTION) {
			return;
		}
		// Reset warnings counter 
		getPrism().getMainLog().resetNumberOfWarnings();
		// Do export...
		handler.export(new ModelExportTask(exportEntity, getChooserFile(), new ModelExportOptions(exportFormat)));
	}

	protected void a_viewBuild(ModelExportEntity exportEntity, ModelExportFormat exportFormat)
	{
		// Reset warnings counter 
		getPrism().getMainLog().resetNumberOfWarnings();
		// Do view...
		handler.export(new ModelExportTask(exportEntity, (File) null, new ModelExportOptions(exportFormat)));
	}

	// 	protected void a_viewStates()
	// 	{
	// 		handler.requestViewStates();
	// 	}

	protected void a_viewCurrentModelBuild()
	{
		handler.requestViewModel();
	}

	protected void a_exportSteadyState(ModelExportFormat exportFormat)
	{
		// Pop up dialog to select file
		int res = JFileChooser.CANCEL_OPTION;
		switch (exportFormat) {
			case MATLAB:
				res = showSaveFileDialog(matlabFilter);
				break;
			case EXPLICIT:
			default:
				res = showSaveFileDialog(textFilter);
				break;
		}
		if (res != JFileChooser.APPROVE_OPTION) {
			return;
		}
		// Reset warnings counter 
		getPrism().getMainLog().resetNumberOfWarnings();
		// Do steady-state
		handler.computeSteadyState(new ModelExportTask(ModelExportEntity.MODEL, getChooserFile(), new ModelExportOptions(exportFormat)));
	}

	protected void a_computeSteadyState()
	{
		// Reset warnings counter 
		getPrism().getMainLog().resetNumberOfWarnings();
		// Do steady-state
		handler.computeSteadyState(new ModelExportTask(ModelExportEntity.MODEL, (File) null, new ModelExportOptions(ModelExportFormat.EXPLICIT)));
	}

	protected void a_exportTransient(ModelExportFormat exportFormat)
	{
		// Get time
		int result = GUITransientTime.requestTime(this.getGUI());
		if (result != GUITransientTime.OK)
			return;
		// Pop up dialog to select file
		int res = JFileChooser.CANCEL_OPTION;
		switch (exportFormat) {
			case MATLAB:
				res = showSaveFileDialog(matlabFilter);
				break;
			case EXPLICIT:
			default:
				res = showSaveFileDialog(textFilter);
				break;
		}
		if (res != JFileChooser.APPROVE_OPTION) {
			return;
		}
		// Reset warnings counter 
		getPrism().getMainLog().resetNumberOfWarnings();
		// Do transient
		handler.computeTransient(GUITransientTime.getTimeSpec(), new ModelExportTask(ModelExportEntity.MODEL, getChooserFile(), new ModelExportOptions(exportFormat)));
	}

	protected void a_computeTransient()
	{
		// Reset warnings counter 
		getPrism().getMainLog().resetNumberOfWarnings();
		// Get time
		int result = GUITransientTime.requestTime(this.getGUI());
		if (result != GUITransientTime.OK)
			return;
		// Do transient
		handler.computeTransient(GUITransientTime.getTimeSpec(), new ModelExportTask(ModelExportEntity.MODEL, (File) null, new ModelExportOptions(ModelExportFormat.EXPLICIT)));
	}

	protected void a_convertToPrismTextModel()
	{
		int cont = doModificationCheck();
		if (cont == CONTINUE && (handler.getModelMode() != GUIMultiModelHandler.PRISM_MODE)) {
			String[] selection = { "Yes", "No", "Cancel" };
			int selectionNo = -1;
			selectionNo = optionPane("WARNING: This is a one way operation. Continue?", "Question", JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE,
					selection, selection[0]);
			switch (selectionNo) {
			case 0:
				handler.convertViewToPRISM();
				break;
			}
		}
	}

	private void setupActions()
	{
		newPRISMModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_newPRISMModel();
			}
		};
		newPRISMModel.putValue(Action.LONG_DESCRIPTION, "Removes the current build, and loads a new model editor in PRISM Text Model mode.");
		//newPRISMModel.putValue(Action.SHORT_DESCRIPTION, "New PRISM Text Model");
		newPRISMModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_P));
		newPRISMModel.putValue(Action.NAME, "PRISM model");
		newPRISMModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFilePrism.png"));
		newPRISMModel.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_N, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));

		newPEPAModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				//modified check do later
				a_newPEPAModel();
			}
		};
		newPEPAModel.putValue(Action.LONG_DESCRIPTION, "Removes the current build, and loads a new model editor in PEPA Text Model mode.");
		//newPEPAModel.putValue(Action.SHORT_DESCRIPTION, "New PEPA Text Model");
		newPEPAModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_E));
		newPEPAModel.putValue(Action.NAME, "PEPA model");
		newPEPAModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFilePepa.png"));

		loadModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				//do actions later
				//System.out.println("realised open action");
				a_openModel();

			}
		};
		loadModel
				.putValue(
						Action.LONG_DESCRIPTION,
						"Brings up a file loading dialogue and loads the file into the editor.  The editor will change mode according to the format of the file.  The loaded file is active for saving.");
		//loadModel.putValue(Action.SHORT_DESCRIPTION, "Open Model");
		loadModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_O));
		loadModel.putValue(Action.NAME, "Open model...");
		//loadModel.putValue(Action.ACTION_COMMAND_KEY,
		loadModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallOpen.png"));
		loadModel.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_O, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));

		reloadModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_reloadModel();
			}
		};
		reloadModel.putValue(Action.LONG_DESCRIPTION, "Reloads the current active file.");
		//reloadModel.putValue(Action.SHORT_DESCRIPTION, "Reload Model");
		reloadModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_R));
		reloadModel.putValue(Action.NAME, "Reload model");
		//loadModel.putValue(Action.ACTION_COMMAND_KEY,
		reloadModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallReload.png"));
		reloadModel.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_R, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));

		saveModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_saveModel();
			}
		};
		saveModel.putValue(Action.LONG_DESCRIPTION,
				"Brings up a file saving dialogue and saves the current text editor to the active file or to a new file.  The saved file becomes active");
		//saveModel.putValue(Action.SHORT_DESCRIPTION, "Save Model");
		saveModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_S));
		saveModel.putValue(Action.NAME, "Save model");
		saveModel.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_S, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
		//saveModel.putValue(Action.ACTION_COMMAND_KEY,
		saveModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallSave.png"));

		saveAsModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_saveModelAs();
			}
		};
		saveAsModel.putValue(Action.LONG_DESCRIPTION,
				"Brings up a file saving dialogue and saves the current text editor to a new file.  The saved file becomes active");
		//saveAsModel.putValue(Action.SHORT_DESCRIPTION, "Save Model As...");
		saveAsModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_A));
		saveAsModel.putValue(Action.NAME, "Save model as...");
		//saveAsModel.putValue(Action.ACTION_COMMAND_KEY,
		saveAsModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallSaveAs.png"));

		parseModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_refreshParseTree();
			}
		};
		parseModel.putValue(Action.LONG_DESCRIPTION, "Forces a parse of the model in the editor.  The parsed description is shown in the model tree.");
		//parseModel.putValue(Action.SHORT_DESCRIPTION, "Parse Model");
		parseModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_P));
		parseModel.putValue(Action.NAME, "Parse model");
		parseModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallParse.png"));
		parseModel.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_F2, 0));
		buildModel = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_build();
			}
		};
		buildModel.putValue(Action.LONG_DESCRIPTION, "Builds the model that has been parsed.");
		//buildModel.putValue(Action.SHORT_DESCRIPTION, "Build Model");
		buildModel.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_B));
		buildModel.putValue(Action.NAME, "Build model");
		buildModel.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallBuild.png"));
		buildModel.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0));

		exportStatesPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.STATES, ModelExportFormat.EXPLICIT);
			}
		};
		exportStatesPlain.putValue(Action.LONG_DESCRIPTION, "Exports the states to a plain text file");
		exportStatesPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_S));
		exportStatesPlain.putValue(Action.NAME, "States");
		exportStatesPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		exportStatesMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.STATES, ModelExportFormat.MATLAB);
			}
		};
		exportStatesMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the states to a Matlab file");
		exportStatesMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_S));
		exportStatesMatlab.putValue(Action.NAME, "States");
		exportStatesMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		exportTransPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.MODEL, ModelExportFormat.EXPLICIT);
			}
		};
		exportTransPlain.putValue(Action.LONG_DESCRIPTION, "Exports the transition matrix to a plain text file");
		exportTransPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_T));
		exportTransPlain.putValue(Action.NAME, "Transition matrix");
		exportTransPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallMatrix.png"));

		exportTransMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.MODEL, ModelExportFormat.MATLAB);
			}
		};
		exportTransMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the transition matrix to a Matlab file");
		exportTransMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_T));
		exportTransMatlab.putValue(Action.NAME, "Transition matrix");
		exportTransMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallMatrix.png"));

		exportTransDot = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.MODEL, ModelExportFormat.DOT);
			}
		};
		exportTransDot.putValue(Action.LONG_DESCRIPTION, "Exports the model to a Dot file");
		exportTransDot.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_D));
		exportTransDot.putValue(Action.NAME, "Dot");
		exportTransDot.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFileDot.png"));

		exportTransUMB = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.MODEL, ModelExportFormat.UMB);
			}
		};
		exportTransUMB.putValue(Action.LONG_DESCRIPTION, "Exports the model to a UMB file");
		exportTransUMB.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_D));
		exportTransUMB.putValue(Action.NAME, "UMB");
		exportTransUMB.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallMatrix.png"));

		exportObsPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.OBSERVATIONS, ModelExportFormat.EXPLICIT);
			}
		};
		exportObsPlain.putValue(Action.LONG_DESCRIPTION, "Exports the observations to a plain text file");
		exportObsPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_O));
		exportObsPlain.putValue(Action.NAME, "Observations");
		exportObsPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		exportObsMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.OBSERVATIONS, ModelExportFormat.MATLAB);
			}
		};
		exportObsMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the observations to a Matlab file");
		exportObsMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_O));
		exportObsMatlab.putValue(Action.NAME, "Observations");
		exportObsMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		exportStateRewardsPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.STATE_REWARDS, ModelExportFormat.EXPLICIT);
			}
		};
		exportStateRewardsPlain.putValue(Action.LONG_DESCRIPTION, "Exports the state rewards vector to a plain text file");
		exportStateRewardsPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_R));
		exportStateRewardsPlain.putValue(Action.NAME, "State rewards");
		exportStateRewardsPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("JMenu .png"));

		exportStateRewardsMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.STATE_REWARDS, ModelExportFormat.MATLAB);
			}
		};
		exportStateRewardsMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the state rewards vector to a Matlab file");
		exportStateRewardsMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_R));
		exportStateRewardsMatlab.putValue(Action.NAME, "State rewards");
		exportStateRewardsMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("JMenu .png"));

		exportTransRewardsPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.STATE_REWARDS, ModelExportFormat.EXPLICIT);
			}
		};
		exportTransRewardsPlain.putValue(Action.LONG_DESCRIPTION, "Exports the transition rewards matrix to a plain text file");
		exportTransRewardsPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_E));
		exportTransRewardsPlain.putValue(Action.NAME, "Transition rewards");
		exportTransRewardsPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallMatrix.png"));

		exportTransRewardsMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.TRANSITION_REWARDS, ModelExportFormat.MATLAB);
			}
		};
		exportTransRewardsMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the transition rewards matrix to a Matlab file");
		exportTransRewardsMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_E));
		exportTransRewardsMatlab.putValue(Action.NAME, "Transition rewards");
		exportTransRewardsMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallMatrix.png"));

		exportLabelsPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.LABELS, ModelExportFormat.EXPLICIT);
			}
		};
		exportLabelsPlain.putValue(Action.LONG_DESCRIPTION, "Exports the model's labels and their satisfying states to a plain text file");
		exportLabelsPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_L));
		exportLabelsPlain.putValue(Action.NAME, "Labels");
		exportLabelsPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		exportLabelsMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportBuildAs(ModelExportEntity.LABELS, ModelExportFormat.MATLAB);
			}
		};
		exportLabelsMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the model's labels and their satisfying states to a Matlab file");
		exportLabelsMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_L));
		exportLabelsMatlab.putValue(Action.NAME, "Labels");
		exportLabelsMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		computeSS = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_computeSteadyState();
			}
		};
		computeSS.putValue(Action.LONG_DESCRIPTION, "Computes steady-state probabilities for the model");
		//computeSS.putValue(Action.SHORT_DESCRIPTION, "Compute steady-state probabilities");
		computeSS.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_S));
		computeSS.putValue(Action.NAME, "Steady-state probabilities");
		computeSS.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallSteadyState.png"));
		computeSS.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_F4, 0));

		computeTr = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_computeTransient();
			}
		};
		computeTr.putValue(Action.LONG_DESCRIPTION, "Computes transient probabilities for the model");
		//computeTr.putValue(Action.SHORT_DESCRIPTION, "Compute transient probabilities");
		computeTr.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_T));
		computeTr.putValue(Action.NAME, "Transient probabilities");
		computeTr.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallClockAnim1.png"));
		computeTr.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_F4, KeyEvent.CTRL_DOWN_MASK));

		exportSSPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportSteadyState(ModelExportFormat.EXPLICIT);
			}
		};
		exportSSPlain.putValue(Action.LONG_DESCRIPTION, "Exports the steady-state probabilities to a plain text file");
		exportSSPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_P));
		exportSSPlain.putValue(Action.NAME, "Plain text file");
		exportSSPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFileText.png"));

		exportSSMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportSteadyState(ModelExportFormat.MATLAB);
			}
		};
		exportSSMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the steady-state probabilities to a Matlab file");
		exportSSMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_M));
		exportSSMatlab.putValue(Action.NAME, "Matlab file");
		exportSSMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFileMatlab.png"));

		exportTrPlain = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportTransient(ModelExportFormat.EXPLICIT);
			}
		};
		exportTrPlain.putValue(Action.LONG_DESCRIPTION, "Exports the transient probabilities to a plain text file");
		exportTrPlain.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_P));
		exportTrPlain.putValue(Action.NAME, "Plain text file");
		exportTrPlain.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFileText.png"));

		exportTrMatlab = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_exportTransient(ModelExportFormat.MATLAB);
			}
		};
		exportTrMatlab.putValue(Action.LONG_DESCRIPTION, "Exports the transient probabilities to a Matlab file");
		exportTrMatlab.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_M));
		exportTrMatlab.putValue(Action.NAME, "Matlab file");
		exportTrMatlab.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFileMatlab.png"));

		viewStates = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_viewBuild(ModelExportEntity.STATES, ModelExportFormat.EXPLICIT);
			}
		};
		viewStates.putValue(Action.LONG_DESCRIPTION, "Print the reachable states to the log");
		viewStates.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_S));
		viewStates.putValue(Action.NAME, "States");
		viewStates.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		viewTrans = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_viewBuild(ModelExportEntity.MODEL, ModelExportFormat.EXPLICIT);
			}
		};
		viewTrans.putValue(Action.LONG_DESCRIPTION, "Print the transition matrix to the log");
		viewTrans.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_T));
		viewTrans.putValue(Action.NAME, "Transition matrix");
		viewTrans.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallMatrix.png"));

		viewObs = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_viewBuild(ModelExportEntity.OBSERVATIONS, ModelExportFormat.EXPLICIT);
			}
		};
		viewObs.putValue(Action.LONG_DESCRIPTION, "Print the observations to the log");
		viewObs.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_S));
		viewObs.putValue(Action.NAME, "Observations");
		viewObs.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		viewStateRewards = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_viewBuild(ModelExportEntity.STATE_REWARDS, ModelExportFormat.EXPLICIT);
			}
		};
		viewStateRewards.putValue(Action.LONG_DESCRIPTION, "Print the state rewards to the log");
		viewStateRewards.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_R));
		viewStateRewards.putValue(Action.NAME, "State rewards");
		viewStateRewards.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		viewTransRewards = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_viewBuild(ModelExportEntity.TRANSITION_REWARDS, ModelExportFormat.EXPLICIT);
			}
		};
		viewTransRewards.putValue(Action.LONG_DESCRIPTION, "Print the transition rewards to the log");
		viewTransRewards.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_E));
		viewTransRewards.putValue(Action.NAME, "Transition rewards");
		viewTransRewards.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallMatrix.png"));

		viewLabels = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_viewBuild(ModelExportEntity.LABELS, ModelExportFormat.EXPLICIT);
			}
		};
		viewLabels.putValue(Action.LONG_DESCRIPTION, "Print the labels and satisfying states to the log");
		viewLabels.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_L));
		viewLabels.putValue(Action.NAME, "Labels");
		viewLabels.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallStates.png"));

		viewPrismCode = new AbstractAction()
		{
			public void actionPerformed(ActionEvent e)
			{
				a_viewCurrentModelBuild();
			}
		};
		viewPrismCode.putValue(Action.LONG_DESCRIPTION, "This shows the parsed model in a text editor.");
		viewPrismCode.putValue(Action.MNEMONIC_KEY, Integer.valueOf(KeyEvent.VK_V));
		viewPrismCode.putValue(Action.NAME, "Parsed PRISM model");
		viewPrismCode.putValue(Action.SMALL_ICON, GUIPrism.getIconFromImage("smallFilePrism.png"));
	}

	//Required to be a GUIPlugin:

	public boolean displaysTab()
	{
		return true;
	}

	public JMenu getMenu()
	{
		return modelMenu;
	}

	public String getTabText()
	{
		return "Model";
	}

	public JToolBar getToolBar()
	{
		return null;
	}

	public String getXMLIDTag()
	{
		return "";
	}

	public Object getXMLSaveTree()
	{
		return null;
	}

	public void loadXML(Object c)
	{

	}

	// if return value is true, event should not be passed on to any more listeners

	public boolean processGUIEvent(GUIEvent e)
	{
		if (e instanceof userinterface.properties.GUIPropertiesEvent) {
			if (e.getID() == userinterface.properties.GUIPropertiesEvent.REQUEST_MODEL_PARSE) {
				handler.requestParse(false);
			}
		} else if (e instanceof GUIClipboardEvent && super.getGUI().getFocussedPlugin() == this) {
			GUIClipboardEvent ce = (GUIClipboardEvent) e;
			if (ce.getComponent() == this) {
				int id = ce.getID();
				if (id == GUIClipboardEvent.CUT)
					handler.cut();
				else if (id == GUIClipboardEvent.COPY)
					handler.copy();
				else if (id == GUIClipboardEvent.PASTE)
					handler.paste();
				else if (id == GUIClipboardEvent.DELETE)
					handler.delete();
				else if (id == GUIClipboardEvent.SELECT_ALL)
					handler.selectAll();
				else if (id == GUIClipboardEvent.UNDO)
					handler.undo();
				else if (id == GUIClipboardEvent.REDO)
					handler.redo();

			}
		} else if (e instanceof GUIComputationEvent) {
			if (e.getID() == GUIComputationEvent.COMPUTATION_START) {
				computing = true;
				doEnables();
				selectionChangeHandler.notifyListeners(new GUIEvent(1));
			} else if (e.getID() == GUIComputationEvent.COMPUTATION_DONE) {
				computing = false;
				doEnables();
				selectionChangeHandler.notifyListeners(new GUIEvent(1));
			} else if (e.getID() == GUIComputationEvent.COMPUTATION_ERROR) {
				computing = false;
				doEnables();
				selectionChangeHandler.notifyListeners(new GUIEvent(1));
			}
		} else if (e instanceof GUIExitEvent) {
			if (e.getID() == GUIExitEvent.REQUEST_EXIT) {
				if (doModificationCheck() != CONTINUE) {
					notifyEventListeners(new GUIExitEvent(GUIExitEvent.CANCEL_EXIT));
					return true;
				}
			}
		}
		return false;
	}

	private JMenu initExportMenu()
	{
		JMenu exportMenu = new JMenu("Export");
		exportMenu.setMnemonic('E');
		exportMenu.setIcon(GUIPrism.getIconFromImage("smallExport.png"));
		JMenu exportPlainMenu = new JMenu("Plain text");
		exportPlainMenu.setMnemonic('P');
		exportPlainMenu.setIcon(GUIPrism.getIconFromImage("smallFileText.png"));
		exportPlainMenu.add(exportTransPlain);
		exportPlainMenu.add(exportStateRewardsPlain);
		exportPlainMenu.add(exportTransRewardsPlain);
		exportPlainMenu.add(exportLabelsPlain);
		exportPlainMenu.add(exportStatesPlain);
		exportPlainMenu.add(exportObsPlain);
		exportMenu.add(exportPlainMenu);
		exportMenu.add(exportTransDot);
		exportMenu.add(exportTransUMB);
		JMenu exportMatlabMenu = new JMenu("Matlab");
		exportMatlabMenu.setMnemonic('M');
		exportMatlabMenu.setIcon(GUIPrism.getIconFromImage("smallFileMatlab.png"));
		exportMatlabMenu.add(exportTransMatlab);
		exportMatlabMenu.add(exportStateRewardsMatlab);
		exportMatlabMenu.add(exportTransRewardsMatlab);
		exportMatlabMenu.add(exportLabelsMatlab);
		exportMatlabMenu.add(exportStatesMatlab);
		exportMatlabMenu.add(exportObsMatlab);
		exportMenu.add(exportMatlabMenu);
		return exportMenu;
	}

	private JMenu initViewMenu()
	{
		JMenu viewMenu = new JMenu("View");
		viewMenu.setMnemonic('V');
		viewMenu.setIcon(GUIPrism.getIconFromImage("smallView.png"));
		viewMenu.add(viewStates);
		viewMenu.add(viewTrans);
		viewMenu.add(viewObs);
		viewMenu.add(viewStateRewards);
		viewMenu.add(viewTransRewards);
		viewMenu.add(viewLabels);
		viewMenu.add(viewPrismCode);
		return viewMenu;
	}

	private JMenu initComputeMenu()
	{
		JMenu computeMenu = new JMenu("Compute");
		computeMenu.setMnemonic('C');
		computeMenu.setIcon(GUIPrism.getIconFromImage("smallCompute.png"));
		computeMenu.add(computeSS);
		computeMenu.add(computeTr);
		return computeMenu;
	}

	private JMenu initComputeExportMenu()
	{
		JMenu computeExportMenu = new JMenu("Compute/export");
		computeExportMenu.setMnemonic('X');
		computeExportMenu.setIcon(GUIPrism.getIconFromImage("smallCompute.png"));
		exportSSMenu = new JMenu("Steady-state probabilities");
		exportSSMenu.setMnemonic('S');
		exportSSMenu.setIcon(GUIPrism.getIconFromImage("smallSteadyState.png"));
		exportSSMenu.add(exportSSPlain);
		exportSSMenu.add(exportSSMatlab);
		computeExportMenu.add(exportSSMenu);
		exportTrMenu = new JMenu("Transient probabilities");
		exportTrMenu.setMnemonic('A');
		exportTrMenu.setIcon(GUIPrism.getIconFromImage("smallClockAnim1.png"));
		exportTrMenu.add(exportTrPlain);
		exportTrMenu.add(exportTrMatlab);
		computeExportMenu.add(exportTrMenu);
		return computeExportMenu;
	}

	private void initComponents()
	{
		setupActions();

		modelMenu = new JMenu("Model");
		exportMenu = initExportMenu();
		viewMenu = initViewMenu();
		computeMenu = initComputeMenu();
		computeExportMenu = initComputeExportMenu();

		JPanel topPanel = new JPanel();
		{
			fileTextField = new JTextField();
			{
				fileTextField.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
				fileTextField.setBorder(new javax.swing.border.EtchedBorder());
				fileTextField.setMinimumSize(new java.awt.Dimension(40, 25));
				fileTextField.setEditable(false);
				fileTextField.setBackground(null);
			}

			//progress = new JProgressBar(0, 100);
			topPanel.setLayout(new BorderLayout());
			handler = new GUIMultiModelHandler(this);
			//topPanel.add(progress, BorderLayout.WEST);
			topPanel.add(fileTextField, BorderLayout.NORTH);
			topPanel.add(handler, BorderLayout.CENTER);
		}

		newMenu = new JMenu("New");
		newMenu.setMnemonic('N');
		newMenu.setIcon(GUIPrism.getIconFromImage("smallNew.png"));
		newMenu.add(newPRISMModel);
		newMenu.add(newPEPAModel);
		modelMenu.add(newMenu);
		modelMenu.add(new JSeparator());
		modelMenu.add(loadModel);
		modelMenu.add(reloadModel);
		modelMenu.add(new JSeparator());
		modelMenu.add(saveModel);
		modelMenu.add(saveAsModel);
		modelMenu.add(new JSeparator());
		modelMenu.setMnemonic(KeyEvent.VK_M);
		modelMenu.add(parseModel);
		modelMenu.add(buildModel);
		modelMenu.add(new JSeparator());

		modelMenu.add(exportMenu);

		modelMenu.add(viewMenu);

		modelMenu.add(computeMenu);

		modelMenu.add(computeExportMenu);

		popup = new JPopupMenu();
		{
			popup.add(parseModel);
			popup.add(buildModel);
			popup.add(viewPrismCode);
		}

		modelFilters = new HashMap<String,FileFilter>();
		modelFilters.put("prism", new FileNameExtensionFilter("PRISM models (*.prism, *.pm, *.nm, *.sm, *.smg)", "prism", "pm", "nm", "sm", "smg")); 
		modelFilters.put("pepa", new FileNameExtensionFilter("PEPA models (*.pepa)", "pepa"));
		staFilters = new HashMap<String,FileFilter>();
		staFilters.put("sta", new FileNameExtensionFilter("State list files (*.sta)", "sta"));
		staFilters.put("txt", new FileNameExtensionFilter("Plain text files (*.txt)", "txt"));
		obsFilters = new HashMap<String,FileFilter>();
		obsFilters.put("obs", new FileNameExtensionFilter("Observations files (*.obs)", "obs"));
		obsFilters.put("txt", new FileNameExtensionFilter("Plain text files (*.txt)", "txt"));
		traFilters = new HashMap<String,FileFilter>();
		traFilters.put("tra", new FileNameExtensionFilter("Transition matrix files (*.tra)", "tra"));
		traFilters.put("txt", new FileNameExtensionFilter("Plain text files (*.txt)", "txt"));
		labFilters = new HashMap<String,FileFilter>();
		labFilters.put("lab", new FileNameExtensionFilter("Label files (*.lab)", "lab"));
		labFilters.put("txt", new FileNameExtensionFilter("Plain text files (*.txt)", "txt"));
		textFilter =  new FileNameExtensionFilter("Plain text files (*.txt)", "txt");
		matlabFilter = new FileNameExtensionFilter("Matlab files (*.m)", "m");
		dotFilter = new FileNameExtensionFilter("Dot files (*.dot)", "dot");
		umbFilter = new FileNameExtensionFilter("UMB files (*.umb)", "dot");

		setLayout(new BorderLayout());
		add(topPanel, BorderLayout.CENTER);

		doEnables();
	}

	public void notifySettings(PrismSettings settings)
	{
		//System.out.println("model notifySettings called");
		handler.notifySettings(settings);

		repaint();
	}

	@Override
	public GUIUndoManager getUndoManager()
	{
		return handler.getUndoManager();
	}

	@Override
	public boolean canDoClipBoardAction(Action action)
	{
		if (computing)
			return false;

		return handler.canDoClipBoardAction(action);
	}

	public AbstractAction getParseModel()
	{
		return parseModel;
	}

	public AbstractAction getBuildModel()
	{
		return buildModel;
	}

	public JMenu getViewMenu()
	{
		return initViewMenu();
	}

	public JMenu getExportMenu()
	{
		return initExportMenu();
	}

	public JMenu getComputeMenu()
	{
		return initComputeMenu();
	}

	public JMenu getComputeExportMenu()
	{
		return initComputeExportMenu();
	}

    public void setGUISimulator(GUISimulator sim)
    {
	this.sim = sim;
    }

    public GUISimulator getGUISimulator()
    {
	return sim;
    }

}
