//==============================================================================
//	
//	Copyright (c) 2002-
//	Authors:
//	* Dave Parker <d.a.parker@cs.bham.ac.uk> (University of Birmingham/Oxford)
//	* Gabriel Santos <gabriel.santos@cs.ox.ac.uk> (University of Oxford)
//
//------------------------------------------------------------------------------
//	
//	This file is part of PRISM.
//	
//	PRISM is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//	
//	PRISM is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//	
//	You should have received a copy of the GNU General Public License
//	along with PRISM; if not, write to the Free Software Foundation,
//	Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//	
//==============================================================================

package symbolic.build;

import java.util.ArrayList;
import java.util.List;
import java.util.HashSet;
import java.util.Iterator;

import jdd.*;
import parser.*;
import parser.ast.*;
import prism.*;
import symbolic.comp.StateModelChecker;
import symbolic.model.GamesModel;
import symbolic.model.ModelSymbolic;
import symbolic.model.ModelVariablesDD;
import symbolic.model.Model;
import symbolic.model.NondetModel;
import symbolic.model.ProbModel;
import symbolic.model.StochModel;

// class to translate a modules description file into an MTBDD model

public class Modules2MTBDD
{
	// Prism object
	private Prism prism;
	
	// StateModelChecker for expression -> MTBDD conversion
	private StateModelChecker expr2mtbdd;
	
	// logs
	private PrismLog mainLog;		// main log

	// ModulesFile object to store syntax tree from parser
	private ModulesFile modulesFile;
	
	// model info
	
	// type
	private ModelType modelType;				// model type (dtmc/mdp/ctmc)
	// modules
	private int numModules;			// number of modules
	private String[] moduleNames;	// module names
	// vars/constants
	private int numVars;			// total number of module variables
	private VarList varList;		// list of module variables
	private Values constantValues;	// values of constants
	// synch info
	private int numSynchs;			// number of synchronisations
	private List<String> synchs;			// synchronisations
	// games info
	private int numPlayers;				// number of players
	// rewards
	private int numRewardStructs;		// number of reward structs
	private String[] rewardStructNames;	// reward struct names
	
	// mtbdd stuff
	
	// dds/dd vars - whole system
	private JDDNode trans;				// transition matrix dd
	private JDDNode range;				// dd giving range for system
	private JDDNode start;				// dd for start state
	private JDDNode stateRewards[];		// dds for state rewards
	private JDDNode transRewards[];		// dds for transition rewards
	private JDDNode transActions;	// dd for transition action labels (MDPs)
	private JDDNode transPerAction[];	// dds for transitions for each action (D/CTMCs)
	private JDDNode transInd;	// dds for independent bits of trans
	private JDDNode transSynch[];	// dds for synch action parts of trans
	private JDDVars allDDRowVars;		// all dd vars (rows)
	private JDDVars allDDColVars;		// all dd vars (cols)
	private JDDVars allDDSynchVars;		// all dd vars (synchronising actions)
	private JDDVars allDDSchedVars;		// all dd vars (scheduling)
	private JDDVars allDDChoiceVars;	// all dd vars (internal non-det.)
	private JDDVars allDDNondetVars;	// all dd vars (all non-det.)
	private JDDVars allDDPlayerVars;	// all dd vars (players)
	// dds/dd vars - globals
	private JDDVars globalDDRowVars;	// dd vars for all globals (rows)
	private JDDVars globalDDColVars;	// dd vars for all globals (cols)
	// dds/dd vars - modules
	private JDDVars[] moduleDDRowVars;	// dd vars for each module (rows)
	private JDDVars[] moduleDDColVars;	// dd vars for each module (cols)
	private JDDNode[] moduleRangeDDs;	// dd giving range for each module
	private JDDNode[] moduleIdentities;	// identity matrix for each module
	// dds/dd vars - variables
	private JDDVars[] varDDRowVars;		// dd vars (row/col) for each module variable
	private JDDVars[] varDDColVars;
	private JDDNode[] varRangeDDs;		// dd giving range for each module variable
	private JDDNode[] varColRangeDDs;	// dd giving range for each module variable (in col vars)
	private JDDNode[] varIdentities;	// identity matrix for each module variable
	// dds/dd vars - nondeterminism
	private JDDNode[] ddSynchVars;		// individual dd vars for synchronising actions
	private JDDNode[] ddSchedVars;		// individual dd vars for scheduling non-det.
	private JDDNode[] ddChoiceVars;		// individual dd vars for local non-det.
	// dds/dd vars - games
	private JDDNode[] ddPlayerVars;		// individual dd vars for each player
	private JDDNode[] ddPlayerCubes;	// cube over dd vars for each player (i.e. one-hot encoding)

	private ModelVariablesDD modelVariables;
	
	// flags for keeping track of which variables have been used
	private boolean[] varsUsed;
	
	// symmetry info
	private boolean doSymmetry;			// use symmetry reduction
	private JDDNode symm; 				// dd of symmetric states
	private JDDNode nonSymms[];			// dds of non-(i,i+1)-symmetric states (i=0...numSymmModules-1)
	private int numModulesBeforeSymm;	// number of modules in the PRISM file before the symmetric ones
	private int numModulesAfterSymm;	// number of modules in the PRISM file after the symmetric ones
	private int numSymmModules;			// number of symmetric components
	
	// hidden option - do we also store each part of the transition matrix separately? (now defunct)
	private boolean storeTransParts = false; 
	// hidden option - do we also store action info for the transition matrix? (supersedes the above)
	private boolean storeTransActions = true;

	/** Flag, tracking whether the model was already constructed (to know how much cleanup we have to do) */
	private boolean modelWasBuilt = false;

	/** Data structure to store mtbdds for a command */
	private static class CommandDDs
	{
		/** BDD for the guard of a command */
		public JDDNode guard;
		/** MTBDD for the updates of a command */
		public JDDNode up;

		/** Constructor, assigning ZERO to guard and up */
		public CommandDDs()
		{
			guard = JDD.Constant(0.0);
			up = JDD.Constant(0.0);
		}

		/** Constructor */
		public CommandDDs(JDDNode guardDD, JDDNode upDD)
		{
			this.guard = guardDD;
			this.up = upDD;
		}

		/** Deref the dds (if not null) */
		public void clear()
		{
			JDD.DerefNonNull(guard);
			JDD.DerefNonNull(up);
		}
	}

	/**
	 * Data structure used to store mtbdds and related info
	 * for an update
	 */
	private static class UpdateDDs
	{
		/** MTBDD for the updates */
		public JDDNode up;

		public UpdateDDs(JDDNode up)
		{
			this.up = up;
		}

		public void clear() {
			JDD.DerefNonNull(up);
		}
	}

	/**
	 * Data structure used to store mtbdds and related info
	 * for some component of the whole model
	 */
	private class ComponentDDs
	{
		/** BDD for guards */
		public JDDNode guards;
		/** MTBDD for transitions */
		public JDDNode trans;
		/** MTBDD for each reward structure */
		public JDDNode rewards[];
		/** minimal index of dd vars used for local nondeterminism */
		public int min;
		/** max index of dd vars used for local nondeterminism */
		public int max;

		public ComponentDDs()
		{
			rewards = new JDDNode[modulesFile.getNumRewardStructs()];
		}
	}

	/**
	 * Data structure used to store mtbdds and related info
	 * for some part of the system definition
	 */
	private static class SystemDDs
	{
		/** The information for independent bit (i.e. tau action) */
		public ComponentDDs ind;
		/** The information for each synchronising action */
		public ComponentDDs[] synchs;
		/** MTBDD for identity matrix */
		public JDDNode id;
		/** Set of all synchs used (syntactic) */
		public HashSet<String> allSynchs;

		public SystemDDs(int n)
		{
			synchs = new ComponentDDs[n];
			allSynchs = new HashSet<String>();
		}
	}
	
	// constructor
	
	public Modules2MTBDD(Prism p, ModulesFile mf)
	{
		prism = p;
		mainLog = p.getMainLog();
		modulesFile = mf;
		// get symmetry reduction info
		String s = prism.getSettings().getString(PrismSettings.PRISM_SYMM_RED_PARAMS);
		doSymmetry = !(s == null || s == "");
	}
	
	@SuppressWarnings("unchecked") // for clone of vector in translate()

	// main method - translate
	public Model translate() throws PrismException
	{
		ModelSymbolic model = null;
		JDDNode tmp, tmp2;
		JDDVars ddv;
		int i;
		
		// get variable info from ModulesFile
		varList = modulesFile.createVarList();
		if (modulesFile.containsUnboundedVariables())
			throw new PrismNotSupportedException("Cannot build a model that contains a variable with unbounded range (try the explicit engine instead)");
		numVars = varList.getNumVars();
		constantValues = modulesFile.getConstantValues();
		
		// get basic system info
		modelType = modulesFile.getModelType();
		moduleNames = modulesFile.getModuleNames();
		numModules = modulesFile.getNumModules();
		synchs = modulesFile.getSynchs();
		numSynchs = synchs.size();
		numPlayers = modulesFile.getNumPlayers();

		// check model type is supported
		if (!(modelType == ModelType.DTMC || modelType == ModelType.MDP || modelType == ModelType.CTMC || modelType == ModelType.SMG)) {
			throw new PrismException("Symbolic construction of " + modelType + "s not supported");
		}
		
		try {
			// allocate dd variables
			allocateDDVars();
			sortDDVars();
			sortIdentities();
			sortRanges();
			
			// create stripped-down StateModelChecker for expression to MTBDD conversions
			expr2mtbdd = new StateModelChecker(prism, varList, allDDRowVars, varDDRowVars, constantValues);
			
			// translate modules file into dd
			translateModules();

			// get rid of any nondet dd variables not needed
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				tmp = JDD.GetSupport(trans);
				tmp = JDD.ThereExists(tmp, allDDRowVars);
				tmp = JDD.ThereExists(tmp, allDDColVars);
				tmp2 = tmp;
				ddv = new JDDVars();
				while (!tmp2.equals(JDD.ONE)) {
					ddv.addVar(JDD.Var(tmp2.getIndex()));
					tmp2 = tmp2.getThen();
				}
				JDD.Deref(tmp);
				allDDNondetVars.derefAll();
				allDDNondetVars = ddv;
			}

			// build game info
			if (modelType == ModelType.SMG) {
				buildDDGame();
			}

// 		// print dd variables actually used (support of trans)
// 		mainLog.print("\nMTBDD variables used (" + allDDRowVars.n() + "r, " + allDDRowVars.n() + "c");
// 		if (type == ModulesFile.NONDETERMINISTIC) mainLog.print(", " + allDDNondetVars.n() + "nd");
// 		mainLog.print("):");
// 		tmp = JDD.GetSupport(trans);
// 		tmp2 = tmp;
// 		while (!tmp2.isConstant()) {
// 			//mainLog.print(" " + tmp2.getIndex() + ":" + ddVarNames.elementAt(tmp2.getIndex()));
// 			mainLog.print(" " + ddVarNames.elementAt(tmp2.getIndex()));
// 			tmp2 = tmp2.getThen();
// 		}
// 		mainLog.println();
// 		JDD.Deref(tmp);

			// Print some info (if extraddinfo flag on)
			if (prism.getExtraDDInfo()) {
				mainLog.print("Transition matrix (pre-reachability): ");
				mainLog.print(JDD.GetNumNodes(trans) + " nodes (");
				mainLog.print(JDD.GetNumTerminals(trans) + " terminal)\n");
			}

			// build bdd for initial state(s)
			buildInitialStates();

			// store reward struct names
			rewardStructNames = new String[numRewardStructs];
			for (i = 0; i < numRewardStructs; i++) {
				rewardStructNames[i] = modulesFile.getRewardStruct(i).getName();
			}

			// create new Model object to be returned
			if (modelType == ModelType.DTMC) {
				model = new ProbModel(trans, start, allDDRowVars, allDDColVars, modelVariables,
				                      varList, varDDRowVars, varDDColVars);
			}
			else if (modelType == ModelType.MDP) {
				model = new NondetModel(trans, start, allDDRowVars, allDDColVars, allDDNondetVars, modelVariables,
									    varList, varDDRowVars, varDDColVars);
			}
			else if (modelType == ModelType.CTMC) {
				model = new StochModel(trans, start, allDDRowVars, allDDColVars, modelVariables,
				                       varList, varDDRowVars, varDDColVars);
			}
			else if (modelType == ModelType.SMG) {
				PlayerInfo playerInfo = new PlayerInfo();
				for (int player = 0; player < numPlayers; player++) {
					playerInfo.addPlayer(modulesFile.getPlayer(player).getName());
				}
				model = new GamesModel(trans, start, allDDRowVars, allDDColVars, allDDNondetVars, modelVariables,
									   varList, varDDRowVars, varDDColVars, allDDPlayerVars, ddPlayerCubes, playerInfo);
			}
			model.setRewards(stateRewards, transRewards, rewardStructNames);
			model.setConstantValues(constantValues);
			modelWasBuilt = true;

			// We also store a copy of the list of action label names
			model.setSynchs(new ArrayList<>(synchs));
		
			// For MDPs, we also store the DDs used to construct the part
			// of the transition matrix that corresponds to each action
			if ((modelType == ModelType.MDP || modelType == ModelType.SMG) && storeTransParts) {
				((NondetModel)model).setTransInd(transInd);
				((NondetModel)model).setTransSynch(transSynch);
			}

			// If required, we also store info about action labels
			if (storeTransActions) {
				// Note: one of these will be null, depending on model type
				// but this is fine: null = none stored.
				if (!(modelType == ModelType.MDP || modelType == ModelType.SMG)) {
					((ProbModel) model).setTransPerAction(transPerAction);
				} else {
					((NondetModel) model).setTransActions(transActions);
				}
			}

			// do reachability (or not)
			if (prism.getDoReach()) {
				mainLog.print("\nComputing reachable states...\n");
				model.doReachability();
				model.filterReachableStates();
			}
			else {
				mainLog.print("\nSkipping reachable state computation.\n");
				model.skipReachability();
				model.filterReachableStates();
			}

			// Print some info (if extraddinfo flag on)
			if (prism.getExtraDDInfo()) {
				mainLog.print("Reach: " + JDD.GetNumNodes(model.getReach()) + " nodes\n");
			}

			// symmetrification
			if (doSymmetry) doSymmetry(model);

			// find/fix any deadlocks
			model.findDeadlocks(prism.getFixDeadlocks());

		} catch (Exception e) {
			// if the model was already built when the exception occurred, clear it.
			if (model != null)
				model.clear();
			throw e;
		} finally {
			// always clean up the Modules2MTBDD variables
			cleanup();
		}

		return model;
	}

	/**
	 * Perform dd cleanup after translate call. 
	 */
	private void cleanup()
	{
		// deref spare dds
		if (globalDDRowVars != null)
			globalDDRowVars.derefAll();
		if (globalDDColVars != null)
			globalDDColVars.derefAll();
		if (moduleDDRowVars != null)
			JDDVars.derefAllArray(moduleDDRowVars);
		if (moduleDDColVars != null)
			JDDVars.derefAllArray(moduleDDColVars);
		JDD.DerefArrayNonNull(moduleIdentities, numModules);
		JDD.DerefArrayNonNull(moduleRangeDDs, numModules);
		JDD.DerefArrayNonNull(varIdentities, numVars);
		JDD.DerefArrayNonNull(varRangeDDs, numVars);
		JDD.DerefArrayNonNull(varColRangeDDs, numVars);
		JDD.DerefNonNull(range);
		JDD.DerefArrayNonNull(ddSynchVars);
		JDD.DerefArrayNonNull(ddSchedVars);
		JDD.DerefArrayNonNull(ddChoiceVars);
		if (allDDSynchVars != null)
			allDDSynchVars.derefAll();
		if (allDDSchedVars != null)
			allDDSchedVars.derefAll();
		if (allDDChoiceVars != null)
			allDDChoiceVars.derefAll();
		JDD.DerefArrayNonNull(ddPlayerVars);

		if (doSymmetry) {
			JDD.Deref(symm);
			JDD.DerefArray(nonSymms, numSymmModules - 1);
		}

		if (!modelWasBuilt) {
			// if the Model object was not yet constructed, we have to do more cleanup
			JDD.DerefNonNull(trans);
			JDD.DerefNonNull(start);
			JDD.DerefArrayNonNull(stateRewards, numRewardStructs);
			JDD.DerefArrayNonNull(transRewards, numRewardStructs);
			JDD.DerefNonNull(transActions);
			JDD.DerefArrayNonNull(transPerAction);
			JDD.DerefNonNull(transInd);
			JDD.DerefArrayNonNull(transSynch);

			if (allDDRowVars != null)
				allDDRowVars.derefAll();
			if (allDDColVars != null)
				allDDColVars.derefAll();
			if (allDDNondetVars != null)
				allDDNondetVars.derefAll();
			if (allDDPlayerVars != null)
				allDDPlayerVars.derefAll();

			JDDVars.derefAllArray(varDDRowVars);
			JDDVars.derefAllArray(varDDColVars);

			if (modelVariables != null)
				modelVariables.clear();
		}

		if (expr2mtbdd != null)
			expr2mtbdd.clearDummyModel();
	}

	// allocate DD vars for system
	// i.e. decide on variable ordering and request variables from CUDD

	private void allocateDDVars()
	{
		int i, j, m, n, last;
		modelVariables = new ModelVariablesDD();

		// player variables (SMG)
		if (modelType == ModelType.SMG) {
			ddPlayerVars = new JDDNode[numPlayers];
			for (int player = 0; player < numPlayers; player++) {
				ddPlayerVars[player] = modelVariables.allocateVariable(modulesFile.getPlayer(player).getName() + ".p");
			}
		}

		switch (prism.getOrdering()) {
		
		case 1:
		// ordering: (a ... a) (s ... s) (l ... l) (r c ... r c)
		
			modelVariables.preallocateExtraActionVariables(prism.getSettings().getInteger(PrismSettings.PRISM_DD_EXTRA_ACTION_VARS));

			// create arrays/etc. first
			
			// nondeterministic variables
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				// synchronizing action variables
				ddSynchVars = new JDDNode[numSynchs];
				// sched nondet vars
				ddSchedVars = new JDDNode[numModules];
				// local nondet vars
				// max num needed = total num of commands in all modules + num of modules
				// (actually, might need more for complex parallel compositions? hmmm...)
				m = numModules;
				for (i = 0; i < numModules; i++) {
					m += modulesFile.getModule(i).getNumCommands();
				}
				ddChoiceVars = new JDDNode[m];
			}
			// module variable (row/col) vars
			varDDRowVars = new JDDVars[numVars];
			varDDColVars = new JDDVars[numVars];
			for (i = 0; i < numVars; i++) {
				varDDRowVars[i] = new JDDVars();
				varDDColVars[i] = new JDDVars();
			}
			
			// now allocate variables

			// allocate synchronizing action variables
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				// allocate vars
				for (i = 0; i < numSynchs; i++) {
					ddSynchVars[i] = modelVariables.allocateVariable(synchs.get(i)+".a");
				}
			}
		
			// allocate scheduling nondet dd variables
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				// allocate vars
				for (i = 0; i < numModules; i++) {
					ddSchedVars[i] = modelVariables.allocateVariable(moduleNames[i] + ".s");
				}
			}
			
			// allocate internal nondet choice dd variables
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				m = ddChoiceVars.length;
				for (i = 0; i < m; i++) {
					ddChoiceVars[i] = modelVariables.allocateVariable("l" + i);
				}
			}
			
			// create a gap in the dd variables
			// this allows to prepend additional row/col vars, e.g. for constructing
			// a product model when doing LTL model checking
			modelVariables.preallocateExtraStateVariables(prism.getSettings().getInteger(PrismSettings.PRISM_DD_EXTRA_STATE_VARS));

			
			// allocate dd variables for module variables (i.e. rows/cols)
			// go through all vars in order (incl. global variables)
			// so overall ordering can be specified by ordering in the input file
			for (i = 0; i < numVars; i++) {
				// get number of dd variables needed
				// (ceiling of log2 of range of variable)
				n = varList.getRangeLogTwo(i);
				// add pairs of variables (row/col)
				for (j = 0; j < n; j++) {
					// new dd row variable
					varDDRowVars[i].addVar(modelVariables.allocateVariable(varList.getName(i) + "." + j));
					// new dd col variable
					varDDColVars[i].addVar(modelVariables.allocateVariable(varList.getName(i) + "'." + j));
				}
			}

			break;
			
		case 2:
		// ordering: (a ... a) (l ... l) (s r c ... r c) (s r c ... r c) ...
	
			modelVariables.preallocateExtraActionVariables(prism.getSettings().getInteger(PrismSettings.PRISM_DD_EXTRA_ACTION_VARS));

			// create arrays/etc. first
			
			// nondeterministic variables
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				// synchronizing action variables
				ddSynchVars = new JDDNode[numSynchs];
				// sched nondet vars
				ddSchedVars = new JDDNode[numModules];
				// local nondet vars: num = total num of commands in all modules + num of modules
				m = numModules;
				for (i = 0; i < numModules; i++) {
					m += modulesFile.getModule(i).getNumCommands();
				}
				ddChoiceVars = new JDDNode[m];
			}
			// module variable (row/col) vars
			varDDRowVars = new JDDVars[numVars];
			varDDColVars = new JDDVars[numVars];
			for (i = 0; i < numVars; i++) {
				varDDRowVars[i] = new JDDVars();
				varDDColVars[i] = new JDDVars();
			}
			
			// now allocate variables
			
			// allocate synchronizing action variables
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				for (i = 0; i < numSynchs; i++) {
					ddSynchVars[i] = modelVariables.allocateVariable(synchs.get(i)+".a");
				}
			}

			// allocate internal nondet choice dd variables
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
				m = ddChoiceVars.length;
				for (i = 0; i < m; i++) {
					ddChoiceVars[i] = modelVariables.allocateVariable("l" + i);
				}
			}

			// TODO: For the other variable order (-o1, used for sparse/hybrid by default,
			// see above), we preallocate a certain number of state variables.
			// For consistency, it would make sense to do the same here. However,
			// we should first do some testing to see if this negatively impacts
			// performance.

			// go through all vars in order (incl. global variables)
			// so overall ordering can be specified by ordering in the input file
			// use 'last' to detect when starting a new module
			last = -1; // globals are -1
			for (i = 0; i < numVars; i++) {
				// if at the start of a module's variables
				// and model is an mdp...
				if ((modelType == ModelType.MDP || modelType == ModelType.SMG) && (last != varList.getModule(i))) {
					// add scheduling dd var(s) (may do multiple ones here if modules have no vars)
					for (j = last+1; j <= varList.getModule(i); j++) {
						ddSchedVars[j] = modelVariables.allocateVariable(moduleNames[j] + ".s");
					}
					// change 'last'
					last = varList.getModule(i);
				}
				// now add row/col dd vars for the variable
				// get number of dd variables needed
				// (ceiling of log2 of range of variable)
				n = varList.getRangeLogTwo(i);
				// add pairs of variables (row/col)
				for (j = 0; j < n; j++) {
					varDDRowVars[i].addVar(modelVariables.allocateVariable(varList.getName(i) + "." + j));
					varDDColVars[i].addVar(modelVariables.allocateVariable(varList.getName(i) + "'." + j));
				}
			}
			// add any remaining scheduling dd var(s) (happens if some modules have no vars)
			if (modelType == ModelType.MDP || modelType == ModelType.SMG) for (j = last+1; j <numModules; j++) {
				ddSchedVars[j] = modelVariables.allocateVariable(moduleNames[j] + ".s");
			}
			break;
			
		default:
			mainLog.printWarning("Invalid MTBDD ordering selected - it's all going to go wrong.");
			break;
		}
		
		// print out all mtbdd variables allocated
//		mainLog.print("\nMTBDD variables:");
//		for (i = 0; i < ddVarNames.size(); i++) {
//			mainLog.print(" (" + i + ")" + ddVarNames.elementAt(i));
//		}
//		mainLog.println();
	}

	// sort out DD variables and the arrays they are stored in
	// (more than one copy of most variables is stored)
			
	private void sortDDVars()
	{
		int i, m;
		
		// put refs for all globals and all vars in each module together
		// create arrays
		globalDDRowVars = new JDDVars();
		globalDDColVars = new JDDVars();
		moduleDDRowVars = new JDDVars[numModules];
		moduleDDColVars = new JDDVars[numModules];
		for (i = 0; i < numModules; i++) {
			moduleDDRowVars[i] = new JDDVars();
			moduleDDColVars[i] = new JDDVars();
		}
		// go thru all variables
		for (i = 0; i < numVars; i++) {
			// check which module it belongs to
			m = varList.getModule(i);
			// if global...
			if (m == -1) {
				globalDDRowVars.copyVarsFrom(varDDRowVars[i]);
				globalDDColVars.copyVarsFrom(varDDColVars[i]);
			}
			// otherwise...
			else {
				moduleDDRowVars[m].copyVarsFrom(varDDRowVars[i]);
				moduleDDColVars[m].copyVarsFrom(varDDColVars[i]);
			}
		}
		
		// put refs for all vars in whole system together
		// create arrays
		allDDRowVars = new JDDVars();
		allDDColVars = new JDDVars();
		if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
			allDDSynchVars = new JDDVars();
			allDDSchedVars = new JDDVars();
			allDDChoiceVars = new JDDVars();
			allDDNondetVars = new JDDVars();
		}
		if (modelType == ModelType.SMG) {
			allDDPlayerVars = new JDDVars();
		}
		// go thru all variables
		for (i = 0; i < numVars; i++) {
			// add to list
			allDDRowVars.copyVarsFrom(varDDRowVars[i]);
			allDDColVars.copyVarsFrom(varDDColVars[i]);
		}
		if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
			// go thru all syncronising action vars
			for (i = 0; i < ddSynchVars.length; i++) {
				// add to list
				allDDSynchVars.addVar(ddSynchVars[i].copy());
				allDDNondetVars.addVar(ddSynchVars[i].copy());
			}
			// go thru all scheduler nondet vars
			for (i = 0; i < ddSchedVars.length; i++) {
				// add to list
				allDDSchedVars.addVar(ddSchedVars[i].copy());
				allDDNondetVars.addVar(ddSchedVars[i].copy());
			}
			// go thru all local nondet vars
			for (i = 0; i < ddChoiceVars.length; i++) {
				// add to list
				allDDChoiceVars.addVar(ddChoiceVars[i].copy());
				allDDNondetVars.addVar(ddChoiceVars[i].copy());
			}
		}
		if (modelType == ModelType.SMG) {
			// go thru all player vars
			for (i = 0; i < ddPlayerVars.length; i++) {
				// add to list
				allDDPlayerVars.addVar(ddPlayerVars[i].copy());
			}
		}
	}
	
	// sort DDs for identities
	
	private void sortIdentities()
	{
		int i, j;
		JDDNode id;
		
		// variable identities
		varIdentities = new JDDNode[numVars];
		for (i = 0; i < numVars; i++) {
			// set each element of the identity matrix
			id = JDD.Constant(0);
			for (j = 0; j < varList.getRange(i); j++) {
				id = JDD.SetMatrixElement(id, varDDRowVars[i], varDDColVars[i], j, j, 1);
			}
			varIdentities[i] = id;
		}
		// module identities
		moduleIdentities = new JDDNode[numModules];
		for (i = 0; i < numModules; i++) {
			// product of identities for vars in module
			id = JDD.Constant(1);
			for (j = 0; j < numVars; j++) {
				if (varList.getModule(j) == i) {
					id = JDD.Apply(JDD.TIMES, id, varIdentities[j].copy());
				}
			}
			moduleIdentities[i] = id;
		}
	}

	// Sort DDs for ranges
	
	private void sortRanges()
	{
		int i;
		
		// initialise raneg for whole system
		range = JDD.Constant(1);
		
		// variable ranges		
		varRangeDDs = new JDDNode[numVars];
		varColRangeDDs = new JDDNode[numVars];
		for (i = 0; i < numVars; i++) {
			// obtain range dd by abstracting from identity matrix
			varRangeDDs[i] = JDD.SumAbstract(varIdentities[i].copy(), varDDColVars[i]);
			// obtain range dd by abstracting from identity matrix
			varColRangeDDs[i] = JDD.SumAbstract(varIdentities[i].copy(), varDDRowVars[i]);
			// build up range for whole system as we go
			range = JDD.Apply(JDD.TIMES, range, varRangeDDs[i].copy());
		}
		// module ranges
		moduleRangeDDs = new JDDNode[numModules];
		for (i = 0; i < numModules; i++) {
			// obtain range dd by abstracting from identity matrix
			moduleRangeDDs[i] = JDD.SumAbstract(moduleIdentities[i].copy(), moduleDDColVars[i]);
		}
	}

	// translate modules decription to dds
	
	private void translateModules() throws PrismException
	{
		SystemFullParallel sys;
		JDDNode tmp;
		int i;
		
		varsUsed = new boolean[numVars];
		
		if (modulesFile.getSystemDefn() == null) {
			sys = new SystemFullParallel();
			for (i = 0; i < numModules; i++) {
				sys.addOperand(new SystemModule(moduleNames[i]));
			}
			translateSystemDefn(sys);
		}
		else {
			translateSystemDefn(modulesFile.getSystemDefn());
		}
		
//		if (type == ModulesFile.PROBABILISTIC) {
//			// divide each row by number of modules
//			trans = JDD.Apply(JDD.DIVIDE, trans, JDD.Constant(numModules));
//		}
		
		// for dtmcs, need to normalise each row to remove local nondeterminism
		if (modelType == ModelType.DTMC) {
			// divide each row by row sum
			tmp = JDD.SumAbstract(trans.copy(), allDDColVars);
			trans = JDD.Apply(JDD.DIVIDE, trans, tmp.copy());
			// also divide action info if needed
			if (storeTransActions) {
				for (i = 0; i < numSynchs + 1; i++) {
					transPerAction[i] = JDD.Apply(JDD.DIVIDE, transPerAction[i], tmp.copy());
				}
			}
			JDD.Deref(tmp);
		}
	}

	// build system according to composition expression
	
	private void translateSystemDefn(SystemDefn sys) throws PrismException
	{
		SystemDDs sysDDs;
		JDDNode tmp, v;
		int i, j, n, max;
		int[] synchMin;
		
		// initialise some values for synchMin
		// (stores min indices of dd vars to use for local nondet)
		synchMin = new int[numSynchs];
		for (i = 0; i < numSynchs; i++) {
			synchMin[i] = 0;
		}
		
		// build system recursively (descend parse tree)
		sysDDs = translateSystemDefnRec(sys, synchMin);
		
		// for the nondeterministic case, add extra mtbdd variables to encode nondeterminism
		if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
			// need to make sure all parts have the same number of dd variables for nondeterminism
			// so we don't generate lots of extra nondeterministic choices
			// first compute max number of variables used
			max = sysDDs.ind.max;
			for (i = 0; i < numSynchs; i++) {
				if (sysDDs.synchs[i].max > max) {
					max = sysDDs.synchs[i].max;
				}
			}
			// check independent bit has this many variables
			if (max > sysDDs.ind.max) {
				tmp = JDD.Constant(1);
				for (i = sysDDs.ind.max; i < max; i++) {
					v = ddChoiceVars[ddChoiceVars.length-i-1];
					JDD.Ref(v);
					tmp = JDD.And(tmp, JDD.Not(v));
				}
				sysDDs.ind.trans = JDD.Apply(JDD.TIMES, sysDDs.ind.trans, tmp);
				sysDDs.ind.max = max;
			}
			// check each synchronous bit has this many variables
			for (i = 0; i < numSynchs; i++) {
				if (max > sysDDs.synchs[i].max) {
					tmp = JDD.Constant(1);
					for (j = sysDDs.synchs[i].max; j < max; j++) {
						v = ddChoiceVars[ddChoiceVars.length-j-1];
						JDD.Ref(v);
						tmp = JDD.And(tmp, JDD.Not(v));
					}
					sysDDs.synchs[i].trans = JDD.Apply(JDD.TIMES, sysDDs.synchs[i].trans, tmp);
					sysDDs.synchs[i].max = max;
				}
			}
			// now add in new mtbdd variables to distinguish between actions
			// independent bit
			tmp = JDD.Constant(1);
			for (i = 0; i < numSynchs; i++) {
				tmp = JDD.And(tmp, JDD.Not(ddSynchVars[i].copy()));
			}
			sysDDs.ind.trans = JDD.Apply(JDD.TIMES, tmp, sysDDs.ind.trans);
			// synchronous bits
			for (i = 0; i < numSynchs; i++) {
				tmp = JDD.Constant(1);
				for (j = 0; j < numSynchs; j++) {
					if (j == i) {
						tmp = JDD.And(tmp, ddSynchVars[j].copy());
					}
					else {
						tmp = JDD.And(tmp, JDD.Not(ddSynchVars[j].copy()));
					}
				}
				sysDDs.synchs[i].trans = JDD.Apply(JDD.TIMES, tmp, sysDDs.synchs[i].trans);
			}
		}
		
		// build state and transition rewards
		computeRewards(sysDDs);
		
		// now, for all model types, transition matrix can be built by summing over all actions
		// also build transition rewards at the same time
		n = modulesFile.getNumRewardStructs();
		trans = sysDDs.ind.trans.copy();
		for (j = 0; j < n; j++) {
			transRewards[j] = sysDDs.ind.rewards[j];
		}
		for (i = 0; i < numSynchs; i++) {
			trans = JDD.Apply(JDD.PLUS, trans, sysDDs.synchs[i].trans.copy());
			for (j = 0; j < n; j++) {
				transRewards[j] = JDD.Apply(JDD.PLUS, transRewards[j], sysDDs.synchs[i].rewards[j]);
			}
		}
		// For D/CTMCs, final rewards are scaled by dividing by total prob/rate for each transition
		// (when individual transition rewards are computed, they are multiplied by individual probs/rates).
		// Need to do this (for D/CTMCs) because transition prob/rate can be the sum of values from
		// several different actions; this gives us the "expected" reward for each transition.
		// (Note, for MDPs, nondeterministic choices are always kept separate so this never occurs.)
		if (modelType != ModelType.MDP && modelType != ModelType.SMG) {
			n = modulesFile.getNumRewardStructs();
			for (j = 0; j < n; j++) {
				transRewards[j] = JDD.Apply(JDD.DIVIDE, transRewards[j], trans.copy());
			}
		}
		
		// For MDPs, we take a copy of the DDs used to construct the part
		// of the transition matrix that corresponds to each action
		if ((modelType == ModelType.MDP || modelType == ModelType.SMG) && storeTransParts) {
			transInd = JDD.ThereExists(JDD.GreaterThan(sysDDs.ind.trans.copy(), 0), allDDColVars);
			transSynch = new JDDNode[numSynchs];
			for (i = 0; i < numSynchs; i++) {
				transSynch[i] = JDD.ThereExists(JDD.GreaterThan(sysDDs.synchs[i].trans.copy(), 0), allDDColVars);
			}
		}
		
		// If required, we also build MTBDD(s) to store the action labels for each transition.
		// The indexing of actions is as follows:
		// independent ("tau", non-action-labelled) transitions have index 0;
		// action-labelled transitions are 1-indexed using the ordering from the model file,
		// i.e. adding 1 to the list of actions from modulesFile.getSynchs().
		// What is actually stored differs for each model type.
		// For MDPs, we just store the action (index) for each state and nondet choice
		// (as an MTBDD 'transActions' over allDDRowVars and allDDNondetVars, with terminals giving index).  
		// For D/CTMCs, we have store to store a copy of the transition matrix for each action
		// (as numSynchs+1 MTBDDs 'transPerAction' over allDDRowVars/allDDColVars, with terminals giving prob/rate)  
		// because one global transition can come from several different actions.
		if (storeTransActions) {
			// Initialise storage to null so we know what we have used
			transActions = null;
			transPerAction = null;
			switch (modelType) {
			case MDP:
			case SMG:
				transActions = JDD.Constant(0);
				// Don't need to store info for independent (action-less) transitions
				// as they are encoded as 0 anyway
				//JDD.Ref(sysDDs.ind.trans);
				//tmp = JDD.ThereExists(JDD.GreaterThan(sysDDs.ind.trans, 0), allDDColVars);
				//transActions = JDD.Apply(JDD.PLUS, transActions, JDD.Apply(JDD.TIMES, tmp, JDD.Constant(1)));
				for (i = 0; i < numSynchs; i++) {
					tmp = JDD.ThereExists(JDD.GreaterThan(sysDDs.synchs[i].trans.copy(), 0), allDDColVars);
					transActions = JDD.Apply(JDD.PLUS, transActions, JDD.Apply(JDD.TIMES, tmp, JDD.Constant(1+i)));
				}
				break;
			case DTMC:
			case CTMC:
				// Just reference DDs and copy them to new array
				transPerAction = new JDDNode[numSynchs + 1];
				transPerAction[0] = sysDDs.ind.trans.copy();
				for (i = 0; i < numSynchs; i++) {
					transPerAction[i + 1] = sysDDs.synchs[i].trans.copy();
				}
				break;
			}
		}
		
		// deref bits of ComponentDD objects - we don't need them any more
		JDD.Deref(sysDDs.ind.guards);
		JDD.Deref(sysDDs.ind.trans);
		for (i = 0; i < numSynchs; i++) {
			JDD.Deref(sysDDs.synchs[i].guards);
			JDD.Deref(sysDDs.synchs[i].trans);
		}
		JDD.Deref(sysDDs.id);
	}

	// recursive part of system composition (descend parse tree)
	
	private SystemDDs translateSystemDefnRec(SystemDefn sys, int[] synchMin) throws PrismException
	{
		SystemDDs sysDDs;
		
		// determine type of current parse tree node
		// and pass to relevant method
		if (sys instanceof SystemModule) {
			sysDDs = translateSystemModule((SystemModule)sys, synchMin);
		}
		else if (sys instanceof SystemBrackets) {
			sysDDs = translateSystemDefnRec(((SystemBrackets)sys).getOperand(), synchMin);
		}
		else if (sys instanceof SystemFullParallel) {
			sysDDs = translateSystemFullParallel((SystemFullParallel)sys, synchMin);
		}
		else if (sys instanceof SystemInterleaved) {
			sysDDs = translateSystemInterleaved((SystemInterleaved)sys, synchMin);
		}
		else if (sys instanceof SystemParallel) {
			sysDDs = translateSystemParallel((SystemParallel)sys, synchMin);
		}
		else if (sys instanceof SystemHide) {
			sysDDs = translateSystemHide((SystemHide)sys, synchMin);
		}
		else if (sys instanceof SystemRename) {
			sysDDs = translateSystemRename((SystemRename)sys, synchMin);
		}
		else if (sys instanceof SystemReference) {
			String name = ((SystemReference) sys).getName();
			SystemDefn sysRef = modulesFile.getSystemDefnByName(name);
			if (sysRef == null)
				throw new PrismLangException("Reference to system " + sys + " which does not exist", sys);
			sysDDs = translateSystemDefnRec(sysRef, synchMin);
		}
		else {
			throw new PrismLangException("Unknown operator in model construction", sys);
		}
		
		return sysDDs;
	}

	// system composition (module)
	
	private SystemDDs translateSystemModule(SystemModule sys, int[] synchMin) throws PrismException
	{
		SystemDDs sysDDs;
		parser.ast.Module module;
		String synch;
		int i, m;
		
		// create object to store result
		sysDDs = new SystemDDs(numSynchs);
		
		// determine which module it is
		m = modulesFile.getModuleIndex(sys.getName());
		module = modulesFile.getModule(m);
		
		// build mtbdd for independent bit
		sysDDs.ind = translateModule(m, module, "", 0);
		// build mtbdd for each synchronising action
		for (i = 0; i < numSynchs; i++) {
			synch = synchs.get(i);
			sysDDs.synchs[i] = translateModule(m, module, synch, synchMin[i]);
		}
		// store identity matrix
		sysDDs.id = moduleIdentities[m].copy();
		
		// store synchs used
		sysDDs.allSynchs.addAll(module.getAllSynchs());
		
		return sysDDs;
	}

	// system composition (full parallel)
	
	private SystemDDs translateSystemFullParallel(SystemFullParallel sys, int[] synchMin) throws PrismException
	{
		SystemDDs sysDDs1, sysDDs2, sysDDs;
		int[] newSynchMin;
		int i, j;
		
		// construct mtbdds for first operand
		sysDDs = translateSystemDefnRec(sys.getOperand(0), synchMin);
		
		// loop through all other operands in the parallel operator
		for (i = 1; i < sys.getNumOperands(); i++) {
		
			// change min to max for potentially synchronising actions
			// store this in new array - old one may still be used elsewhere
			newSynchMin = new int[numSynchs];
			for (j = 0; j < numSynchs; j++) {
				if (sysDDs.allSynchs.contains(synchs.get(j))) {
					newSynchMin[j] = sysDDs.synchs[j].max;
				}
				else {
					newSynchMin[j] = synchMin[j];
				}
			}
			
			// construct mtbdds for next operand
			sysDDs2 = translateSystemDefnRec(sys.getOperand(i), newSynchMin);
			// move sysDDs (operands composed so far) into sysDDs1
			sysDDs1 = sysDDs;
			// we are going to combine sysDDs1 and sysDDs2 and put the result into sysDDs
			sysDDs = new SystemDDs(numSynchs);
			
			// combine mtbdds for independent bit
			sysDDs.ind = translateNonSynchronising(sysDDs1.ind, sysDDs2.ind, sysDDs1.id, sysDDs2.id);
			
			// combine mtbdds for each synchronising action
			for (j = 0; j < numSynchs; j++) {
				// if one operand does not use this action,
				// do asynchronous parallel composition
				if ((sysDDs1.allSynchs.contains(synchs.get(j))?1:0) + (sysDDs2.allSynchs.contains(synchs.get(j))?1:0) == 1) {
					sysDDs.synchs[j] = translateNonSynchronising(sysDDs1.synchs[j], sysDDs2.synchs[j], sysDDs1.id, sysDDs2.id);
				}
				else {
					sysDDs.synchs[j] = translateSynchronising(sysDDs1.synchs[j], sysDDs2.synchs[j]);
				}
			}
			
			// compute identity
			sysDDs.id = JDD.Apply(JDD.TIMES, sysDDs1.id, sysDDs2.id);
			
			// combine lists of synchs
			sysDDs.allSynchs.addAll(sysDDs1.allSynchs);
			sysDDs.allSynchs.addAll(sysDDs2.allSynchs);
		}
		
		return sysDDs;
	}

	// system composition (interleaved)
		
	private SystemDDs translateSystemInterleaved(SystemInterleaved sys, int[] synchMin) throws PrismException
	{
		SystemDDs sysDDs1, sysDDs2, sysDDs;
		int i, j;
	
		// construct mtbdds for first operand
		sysDDs = translateSystemDefnRec(sys.getOperand(0), synchMin);
		
		// loop through all other operands in the parallel operator
		for (i = 1; i < sys.getNumOperands(); i++) {
		
			// construct mtbdds for next operand
			sysDDs2 = translateSystemDefnRec(sys.getOperand(i), synchMin);
			// move sysDDs (operands composed so far) into sysDDs1
			sysDDs1 = sysDDs;
			// we are going to combine sysDDs1 and sysDDs2 and put the result into sysDDs
			sysDDs = new SystemDDs(numSynchs);
			
			// combine mtbdds for independent bit
			sysDDs.ind = translateNonSynchronising(sysDDs1.ind, sysDDs2.ind, sysDDs1.id, sysDDs2.id);
			
			// combine mtbdds for each synchronising action
			for (j = 0; j < numSynchs; j++) {
				sysDDs.synchs[j] = translateNonSynchronising(sysDDs1.synchs[j], sysDDs2.synchs[j], sysDDs1.id, sysDDs2.id);
			}
			
			// compute identity
			sysDDs.id = JDD.Apply(JDD.TIMES, sysDDs1.id, sysDDs2.id);
			
			// combine lists of synchs
			sysDDs.allSynchs.addAll(sysDDs1.allSynchs);
			sysDDs.allSynchs.addAll(sysDDs2.allSynchs);
		}
		
		return sysDDs;
	}

	// system composition (parallel over actions)
	
	private SystemDDs translateSystemParallel(SystemParallel sys, int[] synchMin) throws PrismException
	{
		SystemDDs sysDDs1, sysDDs2, sysDDs;
		boolean[] synchBool;
		int[] newSynchMin;
		int i;
		
		// go thru all synchronising actions and decide if we will synchronise on each one
		synchBool = new boolean[numSynchs];
		for (i = 0; i < numSynchs; i++) {
			synchBool[i] = sys.containsAction(synchs.get(i));
		}
		
		// construct mtbdds for first operand
		sysDDs1 = translateSystemDefnRec(sys.getOperand1(), synchMin);
		
		// change min to max for synchronising actions
		// store this in new array - old one may still be used elsewhere
		newSynchMin = new int[numSynchs];
		for (i = 0; i < numSynchs; i++) {
			if (synchBool[i]) {
				newSynchMin[i] = sysDDs1.synchs[i].max;
			}
			else {
				newSynchMin[i] = synchMin[i];
			}
		}
		
		// construct mtbdds for second operand
		sysDDs2 = translateSystemDefnRec(sys.getOperand2(), newSynchMin);
		
		// create object to store mtbdds
		sysDDs = new SystemDDs(numSynchs);
		
		// combine mtbdds for independent bit
		sysDDs.ind = translateNonSynchronising(sysDDs1.ind, sysDDs2.ind, sysDDs1.id, sysDDs2.id);
		
		// combine mtbdds for each synchronising action
		for (i = 0; i < numSynchs; i++) {
			if (synchBool[i]) {
				sysDDs.synchs[i] = translateSynchronising(sysDDs1.synchs[i], sysDDs2.synchs[i]);
			}
			else {
				sysDDs.synchs[i] = translateNonSynchronising(sysDDs1.synchs[i], sysDDs2.synchs[i], sysDDs1.id, sysDDs2.id);
			}
		}
		
		// combine mtbdds for identity matrices
		sysDDs.id = JDD.Apply(JDD.TIMES, sysDDs1.id, sysDDs2.id);
		
		// combine lists of synchs
		sysDDs.allSynchs.addAll(sysDDs1.allSynchs);
		sysDDs.allSynchs.addAll(sysDDs2.allSynchs);
		
		return sysDDs;
	}
	
	// system composition (hide)
	
	private SystemDDs translateSystemHide(SystemHide sys, int[] synchMin) throws PrismException
	{
		SystemDDs sysDDs1, sysDDs;
		int[] newSynchMin;
		int i;
		
		// reset synchMin to 0 for actions to be hidden
		// store this in new array - old one may still be used elsewhere
		newSynchMin = new int[numSynchs];
		for (i = 0; i < numSynchs; i++) {
			if (sys.containsAction(synchs.get(i))) {
				newSynchMin[i] = 0;
			}
			else {
				newSynchMin[i] = synchMin[i];
			}
		}
		
		// construct mtbdds for operand
		sysDDs1 = translateSystemDefnRec(sys.getOperand(), newSynchMin);
		
		// create object to store mtbdds for result
		sysDDs = new SystemDDs(numSynchs);
		
		// copy across independent bit
		sysDDs.ind = sysDDs1.ind;
		
		// go thru all synchronising actions
		for (i = 0; i < numSynchs; i++) {
			
			// if the action is in the set to be hidden, hide it...
			// note that it doesn't matter if an action is included more than once in the set
			// (although this would be picked up during the syntax check anyway)
			if (sys.containsAction(synchs.get(i))) {
				
				// move these transitions into the independent bit
				sysDDs.ind = combineComponentDDs(sysDDs.ind, sysDDs1.synchs[i]);
				
				// create empty mtbdd for action
				sysDDs.synchs[i] = new ComponentDDs();
				sysDDs.synchs[i].guards = JDD.Constant(0);
				sysDDs.synchs[i].trans = JDD.Constant(0);
				sysDDs.synchs[i].min = 0;
				sysDDs.synchs[i].max = 0;
			}
			// otherwise just copy it across
			else {
				sysDDs.synchs[i] = sysDDs1.synchs[i];
			}
		}
		
		// copy identity too
		sysDDs.id = sysDDs1.id;
		
		// modify list of synchs
		sysDDs.allSynchs.addAll(sysDDs1.allSynchs);
		for (i = 0; i < sys.getNumActions(); i++) {
			sysDDs.allSynchs.remove(sys.getAction(i));
		}
		
		return sysDDs;
	}

	// system composition (rename)
	
	private SystemDDs translateSystemRename(SystemRename sys, int[] synchMin) throws PrismException
	{
		SystemDDs sysDDs1, sysDDs;
		int[] newSynchMin;
		int i, j;
		String s;
		Iterator<String> iter;
		
		// swap some values in synchMin due to renaming
		// store this in new array - old one may still be used elsewhere
		newSynchMin = new int[numSynchs];
		for (i = 0; i < numSynchs; i++) {
			// find out what this action is renamed to
			// (this may be itself, i.e. it's not renamed)
			s = sys.getNewName(synchs.get(i));
			j = synchs.indexOf(s);
			if (j == -1) {
				throw new PrismLangException("Invalid action name \"" + s + "\" in renaming", sys);
			}
			newSynchMin[i] = synchMin[j];
		}
		
		// construct mtbdds for operand
		sysDDs1 = translateSystemDefnRec(sys.getOperand(), newSynchMin);
		
		// create object to store mtbdds for result
		sysDDs = new SystemDDs(numSynchs);
		
		// copy across independent bit
		sysDDs.ind = sysDDs1.ind;
		
		// initially there are no mtbdds in result
		for (i = 0; i < numSynchs; i++) {
			sysDDs.synchs[i] = new ComponentDDs();
			sysDDs.synchs[i].guards = JDD.Constant(0);
			sysDDs.synchs[i].trans = JDD.Constant(0);
			sysDDs.synchs[i].min = 0;
			sysDDs.synchs[i].max = 0;
		}
		
		// go thru all synchronising actions
		for (i = 0; i < numSynchs; i++) {
		
			// find out what this action is renamed to
			// (this may be itself, i.e. it's not renamed)
			// then add it to result
			s = sys.getNewName(synchs.get(i));
			j = synchs.indexOf(s);
			if (j == -1) {
				throw new PrismLangException("Invalid action name \"" + s + "\" in renaming", sys);
			}
			sysDDs.synchs[j] = combineComponentDDs(sysDDs.synchs[j], sysDDs1.synchs[i]);
		}
		
		// copy identity too
		sysDDs.id = sysDDs1.id;
		
		// modify list of synchs
		iter = sysDDs1.allSynchs.iterator();
		while (iter.hasNext()) {
			sysDDs.allSynchs.add(sys.getNewName(iter.next()));
		}
		
		return sysDDs;
	}

	private ComponentDDs translateSynchronising(ComponentDDs compDDs1, ComponentDDs compDDs2) throws PrismException
	{
		ComponentDDs compDDs;
		
		// create object to store result
		compDDs = new ComponentDDs();
		
		// combine parts synchronously
		// first guards
		JDD.Ref(compDDs1.guards);
		JDD.Ref(compDDs2.guards);
		compDDs.guards = JDD.And(compDDs1.guards, compDDs2.guards);
		// then transitions
		JDD.Ref(compDDs1.trans);
		JDD.Ref(compDDs2.trans);
		compDDs.trans = JDD.Apply(JDD.TIMES, compDDs1.trans, compDDs2.trans);
		// compute new min/max
		compDDs.min = (compDDs1.min < compDDs2.min) ? compDDs1.min : compDDs2.min;
		compDDs.max = (compDDs1.max > compDDs2.max) ? compDDs1.max : compDDs2.max;
		
		// deref old stuff
		JDD.Deref(compDDs1.guards);
		JDD.Deref(compDDs2.guards);
		JDD.Deref(compDDs1.trans);
		JDD.Deref(compDDs2.trans);
		
		return compDDs;
	}

	private ComponentDDs translateNonSynchronising(ComponentDDs compDDs1, ComponentDDs compDDs2, JDDNode id1, JDDNode id2) throws PrismException
	{
		ComponentDDs compDDs;
		
		// add identities to mtbdds for transitions
		JDD.Ref(id2);
		compDDs1.trans = JDD.Apply(JDD.TIMES, compDDs1.trans, id2);
		JDD.Ref(id1);
		compDDs2.trans = JDD.Apply(JDD.TIMES, compDDs2.trans, id1);
		
		compDDs = combineComponentDDs(compDDs1, compDDs2);
		
		return compDDs;
	}

	private ComponentDDs combineComponentDDs(ComponentDDs compDDs1, ComponentDDs compDDs2) throws PrismException
	{
		ComponentDDs compDDs;
		JDDNode tmp, v;
		int i;
		
		// create object to store result
		compDDs = new ComponentDDs();
		
		// if no nondeterminism - just add
		if (modelType != ModelType.MDP && modelType != ModelType.SMG) {
			compDDs.guards = JDD.Or(compDDs1.guards, compDDs2.guards);
			compDDs.trans = JDD.Apply(JDD.PLUS, compDDs1.trans, compDDs2.trans);
			compDDs.min = 0;
			compDDs.max = 0;
		}
		// if there's nondeterminism, but one part is empty, it's also easy
		else if (compDDs1.trans.equals(JDD.ZERO)) {
			JDD.Deref(compDDs1.guards);
			compDDs.guards = compDDs2.guards;
			JDD.Deref(compDDs1.trans);
			compDDs.trans = compDDs2.trans;
			compDDs.min = compDDs2.min;
			compDDs.max = compDDs2.max;
		}
		else if (compDDs2.trans.equals(JDD.ZERO)) {
			JDD.Deref(compDDs2.guards);
			compDDs.guards = compDDs1.guards;
			JDD.Deref(compDDs2.trans);
			compDDs.trans = compDDs1.trans;
			compDDs.min = compDDs1.min;
			compDDs.max = compDDs1.max;
		}
		// otherwise, it's a bit more complicated...
		else {
			// make sure two bits have the same number of dd variables for nondeterminism
			// (so we don't generate lots of extra nondeterministic choices)
			if (compDDs1.max > compDDs2.max) {
				tmp = JDD.Constant(1);
				for (i = compDDs2.max; i < compDDs1.max; i++) {
					v = ddChoiceVars[ddChoiceVars.length-i-1];
					JDD.Ref(v);
					tmp = JDD.And(tmp, JDD.Not(v));
				}
				compDDs2.trans = JDD.Apply(JDD.TIMES, compDDs2.trans, tmp);
				compDDs2.max = compDDs1.max;
			}
			else if (compDDs2.max > compDDs1.max) {
				tmp = JDD.Constant(1);
				for (i = compDDs1.max; i < compDDs2.max; i++) {
					v = ddChoiceVars[ddChoiceVars.length-i-1];
					JDD.Ref(v);
					tmp = JDD.And(tmp, JDD.Not(v));
				}
				compDDs1.trans = JDD.Apply(JDD.TIMES, compDDs1.trans, tmp);
				compDDs1.max = compDDs2.max;
			}
			// and then combine
			if (ddChoiceVars.length-compDDs1.max-1 < 0)
				throw new PrismException("Insufficient BDD variables allocated for nondeterminism - please report this as a bug. Thank you");
			v = ddChoiceVars[ddChoiceVars.length-compDDs1.max-1];
			compDDs.guards = JDD.Or(compDDs1.guards, compDDs2.guards);
			JDD.Ref(v);
			compDDs.trans = JDD.ITE(v, compDDs2.trans, compDDs1.trans);
			compDDs.min = compDDs1.min;
			compDDs.max = compDDs1.max+1;
		}
		
		return compDDs;
	}
	
	// translate a single module to a dd
	// for a given synchronizing action ("" = none)
	
	private ComponentDDs translateModule(int m, parser.ast.Module module, String synch, int synchMin) throws PrismException
	{
		ComponentDDs compDDs;
		CommandDDs[] commandsDDs;
		Command command;
		int l, numCommands;
		boolean match;
		
		// get number of commands and set up array accordingly
		numCommands = module.getNumCommands();
		commandsDDs = new CommandDDs[numCommands];

		// translate guard/updates for each command of the module
		for (l = 0; l < numCommands; l++) {
			command = module.getCommand(l);
			// check if command matches requested synch
			match = false;
			if (synch == "") {
				if (command.getSynch() == "") match = true;
			}
			else {
				if (command.getSynch().equals(synch)) match = true;
			}
			// if so translate
			if (match) {
				// store in array
				commandsDDs[l] = translateCommand(m, module, l, command);
			}
			// otherwise use 0
			else {
				commandsDDs[l] = new CommandDDs();
			}
		}

		// combine guard/updates dds for each command
		if (modelType == ModelType.DTMC) {
			compDDs = combineCommandsProb(m, commandsDDs);
		} else if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
			compDDs = combineCommandsNondet(m, commandsDDs, synchMin);
		} else if (modelType == ModelType.CTMC) {
			compDDs = combineCommandsStoch(m, commandsDDs);
		} else {
			throw new PrismException("Unknown model type");
		}

		// deref guards/updates
		for (CommandDDs c : commandsDDs) {
			c.clear();
		}

		return compDDs;
	}

	/**
	 * Translate a command to a CommandDDs.
	 * <br>[ REFs: <i>result</i> ]
	 */
	private CommandDDs translateCommand(int m, parser.ast.Module module, int l, Command command) throws PrismException
	{
		JDDNode guardDD, upDD = null;
		// translate guard
		guardDD = translateExpression(command.getGuard());
		guardDD = JDD.Times(guardDD, range.copy());
		// for an SMG, check which player owns the action/module and add player encoding
		if (modelType == ModelType.SMG) {
			int player;
			if (command.getSynch().isEmpty()) {
				player = modulesFile.getPlayerForModule(command.getParent().getName());
			} else {
				player = modulesFile.getPlayerForAction(command.getSynch());
			}
			if (player != -1) {
				guardDD = JDD.Apply(JDD.TIMES, guardDD, ddPlayerVars[player].copy());
			} else {
				for (int p = 0; p < numPlayers; p++) {
					guardDD = JDD.Apply(JDD.TIMES, guardDD, JDD.Not(ddPlayerVars[p].copy()));
				}
			}
		}
		// check for false guard
		if (guardDD.equals(JDD.ZERO)) {
			// display a warning (unless guard is "false", in which case was probably intentional
			if (!Expression.isFalse(command.getGuard())) {
				String s = "Guard for command " + (l+1) + " of module \"" + module.getName() + "\" is never satisfied.";
				mainLog.printWarning(s);
			}
			// no point bothering to compute the mtbdds for the update
			// if the guard is never satisfied
			upDD = JDD.Constant(0);
		}
		else {
			// translate updates and do some checks on probs/rates
			UpdateDDs up = null;
			try {
				up = translateUpdates(m, l, command.getUpdates(), (command.getSynch()=="")?false:true, guardDD);
				up.up = JDD.Times(up.up, guardDD.copy());
				upDD = up.up.copy();
				checkCommandProbRates(m, module, l, command, guardDD, upDD);
			} catch (Throwable e) {
				JDD.DerefNonNull(guardDD, upDD);
				throw e;
			} finally {
				if (up != null) {
					up.clear();
				}
			}
		}

		return new CommandDDs(guardDD, upDD);
	}

	/**
	 * Check the probabilities/rate of a command for errors (in which case an exception is thrown).
	 * <br>[ REFS: <i>none</i>, DEREFS: <i>none</i> ]
	 * @param m the module index
	 * @param module the Module AST element
	 * @param l the command index (inside the module)
	 * @param command the Command AST element
	 * @param guardDD the guard dd
	 * @param upDD the update dd
	 */
	private void checkCommandProbRates(int m, parser.ast.Module module, int l, Command command, JDDNode guardDD, JDDNode upDD) throws PrismLangException
	{
		// are all probs/rates non-negative?
		double dmin = JDD.FindMin(upDD);
		if (dmin < 0) {
			String s = (modelType == ModelType.CTMC) ? "Rates" : "Probabilities";
			s += " in command " + (l+1) + " of module \"" + module.getName() + "\" are negative";
			s += " (" + dmin + ") for some states.\n";
			s += "Perhaps the guard needs to be strengthened";
			throw new PrismLangException(s, command);
		}
		// only do remaining checks if 'doprobchecks' flag is set
		if (prism.getDoProbChecks()) {
			// sum probs/rates in updates
			JDDNode tmp = JDD.SumAbstract(upDD.copy(), moduleDDColVars[m]);
			tmp = JDD.SumAbstract(tmp, globalDDColVars);
			// put 1s in for sums which are not covered by this guard
			tmp = JDD.ITE(guardDD.copy(), tmp, JDD.Constant(1));
			// compute min/max sums
			dmin = JDD.FindMin(tmp);
			double dmax = JDD.FindMax(tmp);
			// check sums for NaNs
			if (Double.isNaN(dmin) || Double.isNaN(dmax)) {
				JDD.Deref(tmp);
				String s = (modelType == ModelType.CTMC) ? "Rates" : "Probabilities";
				s += " in command " + (l+1) + " of module \"" + module.getName() + "\" have errors (NaN) for some states. ";
				s += "Check for zeros in divide or modulo operations. ";
				s += "Perhaps the guard needs to be strengthened";
				throw new PrismLangException(s, command);
			}
			// check min sums - 1 (ish) for dtmcs/mdps, 0 for ctmcs
			if (modelType != ModelType.CTMC && !PrismUtils.doublesAreEqual(dmin, 1.0)) {
				JDD.Deref(tmp);
				String s = "Probabilities in command " + (l+1) + " of module \"" + module.getName() + "\" sum to less than one";
				s += " (e.g. " + dmin + ") for some states. ";
				s += "Perhaps some of the updates give out-of-range values. ";
				s += "One possible solution is to strengthen the guard";
				throw new PrismLangException(s, command);
			}
			if (modelType == ModelType.CTMC && dmin <= 0) {
				JDD.Deref(tmp);
				// note can't sum to less than zero - already checked for negative rates above
				String s = "Rates in command " + (l+1) + " of module \"" + module.getName() + "\" sum to zero for some states. ";
				s += "Perhaps some of the updates give out-of-range values. ";
				s += "One possible solution is to strengthen the guard";
				throw new PrismLangException(s, command);
			}
			// check max sums - 1 (ish) for dtmcs/mdps, infinity for ctmcs
			if (modelType != ModelType.CTMC && !PrismUtils.doublesAreEqual(dmax, 1.0)) {
				JDD.Deref(tmp);
				String s = "Probabilities in command " + (l+1) + " of module \"" + module.getName() + "\" sum to more than one";
				s += " (e.g. " + dmax + ") for some states. ";
				s += "Perhaps the guard needs to be strengthened";
				throw new PrismLangException(s, command);
			}
			if (modelType == ModelType.CTMC && Double.isInfinite(dmax)) {
				JDD.Deref(tmp);
				String s = "Rates in command " + (l+1) + " of module \"" + module.getName() + "\" sum to infinity for some states. ";
				s += "Perhaps the guard needs to be strengthened";
				throw new PrismLangException(s, command);
			}
			JDD.Deref(tmp);
		}
	}

	/**
	 * Go thru guard/updates dds for all commands of a prob. module and combine.
	 * Also check for any guard overlaps, etc...
	 * <br>[ REFS: <i>result</i>, DEREFS: <i>none</i> ]
	 * @param m the module index
	 * @param commandsDDs array of command dds (guard and updates)
	 */
	private ComponentDDs combineCommandsProb(int m, CommandDDs[] commandsDDs)
	{
		ComponentDDs compDDs;
		int i;
		JDDNode covered, transDD, tmp;
		
		// create object to return result
		compDDs = new ComponentDDs();
		
		// use 'transDD' to build up MTBDD for transitions
		transDD = JDD.Constant(0);
		// use 'covered' to track states covered by guards
		covered = JDD.Constant(0);
		// loop thru commands...
		int numCommands = commandsDDs.length;
		for (i = 0; i < numCommands; i++) {
			JDDNode guardDD = commandsDDs[i].guard;
			JDDNode upDD = commandsDDs[i].up;

			// do nothing if guard is empty
			if (guardDD.equals(JDD.ZERO)) {
				continue;
			}
			// check if command overlaps with previous ones
			tmp = JDD.And(guardDD.copy(), covered.copy());
			if (!(tmp.equals(JDD.ZERO))) {
				// if so, output a warning (but carry on regardless)
				mainLog.printWarning("Guard for command " + (i+1) + " of module \""
					+ moduleNames[m] + "\" overlaps with previous commands.");
			}
			JDD.Deref(tmp);
			// add this command's guard to 'covered'
			covered = JDD.Or(covered, guardDD.copy());
			// add transitions
			transDD = JDD.Plus(transDD, JDD.Times(guardDD.copy(), upDD.copy()));
		}
		
		// store result
		compDDs.guards = covered;
		compDDs.trans = transDD;
		compDDs.min = 0;
		compDDs.max = 0;
		
		return compDDs;
	}

	/**
	 * Go thru guard/updates dds for all commands of a stoch. module and combine.
	 * <br>[ REFS: <i>result</i>, DEREFS: <i>none</i> ]
	 * @param m the module index
	 * @param commandsDDs array of command dds (guard and updates)
	 */
	private ComponentDDs combineCommandsStoch(int m, CommandDDs[] commandsDDs)
	{
		ComponentDDs compDDs;
		int i;
		JDDNode covered, transDD;
		
		// create object to return result
		compDDs = new ComponentDDs();
		
		// use 'transDD 'to build up MTBDD for transitions
		transDD = JDD.Constant(0);
		// use 'covered' to track states covered by guards
		covered = JDD.Constant(0);
		
		// loop thru commands...
		int numCommands = commandsDDs.length;
		for (i = 0; i < numCommands; i++) {
			JDDNode guardDD = commandsDDs[i].guard;
			JDDNode upDD = commandsDDs[i].up;

			// do nothing if guard is empty
			if (guardDD.equals(JDD.ZERO)) {
				continue;
			}
			// add this command's guard to 'covered'
			covered = JDD.Or(covered, guardDD.copy());
			// add transitions
			transDD = JDD.Plus(transDD, JDD.Times(guardDD.copy(), upDD.copy()));
		}
		
		// store result
		compDDs.guards = covered;
		compDDs.trans = transDD;
		compDDs.min = 0;
		compDDs.max = 0;
		
		return compDDs;
	}

	/**
	 * Go thru guard/updates dds for all commands of a non-det. module,
	 * work out guard overlaps and sort out non determinism accordingly.
	 * (non recursive version)
	 * <br>[ REFS: <i>result</i>, DEREFS: <i>none</i> ]
	 * @param m the module index
	 * @param commandsDDs array of command dds (guard and updates)
	 * @param synchMin the minimal synch variable that can be used for this module
	 */
	private ComponentDDs combineCommandsNondet(int m, CommandDDs[] commandsDDs, int synchMin) throws PrismException
	{
		ComponentDDs compDDs;
		int i, j, k, maxChoices, numDDChoiceVarsUsed;
		JDDNode covered, transDD, overlaps, equalsi, tmp, tmp2, tmp3;
		JDDNode[] transDDbits, frees;
		JDDVars ddChoiceVarsUsed;
		
		// create object to return result
		compDDs = new ComponentDDs();
		
		// use 'transDD' to build up MTBDD for transitions
		transDD = JDD.Constant(0);
		// use 'covered' to track states covered by guards
		covered = JDD.Constant(0);

		int numCommands = commandsDDs.length;

		// find overlaps in guards by adding them all up
		overlaps = JDD.Constant(0);
		for (i = 0; i < numCommands; i++) {
			JDDNode guardDD = commandsDDs[i].guard;
			overlaps = JDD.Plus(overlaps, guardDD.copy());
			// compute bdd of all guards at same time
			covered = JDD.Or(covered, guardDD.copy());
		}
		
		// find the max number of overlaps
		// (i.e. max number of nondet. choices)
		maxChoices = (int)Math.round(JDD.FindMax(overlaps));
		
		// if all the guards were false, we're done already
		if (maxChoices == 0) {
			compDDs.guards = covered;
			compDDs.trans = transDD;
			compDDs.min = synchMin;
			compDDs.max = synchMin;
			JDD.Deref(overlaps);
			return compDDs;
		}
		
		// likewise, if there are no overlaps, it's also pretty easy
		if (maxChoices == 1) {
			// add up dds for all commands
			for (i = 0; i < numCommands; i++) {
				JDDNode guardDD = commandsDDs[i].guard;
				JDDNode upDD = commandsDDs[i].up;
				// add up transitions
				transDD = JDD.Plus(transDD, JDD.Times(guardDD.copy(), upDD.copy()));
			}
			compDDs.guards = covered;
			compDDs.trans = transDD;
			compDDs.min = synchMin;
			compDDs.max = synchMin;
			JDD.Deref(overlaps);	
			return compDDs;
		}
		
		// otherwise, it's a bit more complicated...
		
		// first, calculate how many dd vars will be needed
		numDDChoiceVarsUsed = (int)Math.ceil(PrismUtils.log2(maxChoices));
		
		// select the variables we will use and put them in a JDDVars
		ddChoiceVarsUsed = new JDDVars();
		for (i = 0; i < numDDChoiceVarsUsed; i++) {
			if (ddChoiceVars.length-synchMin-numDDChoiceVarsUsed+i < 0)
				throw new PrismException("Insufficient BDD variables allocated for nondeterminism - please report this as a bug. Thank you.");
			ddChoiceVarsUsed.addVar(ddChoiceVars[ddChoiceVars.length-synchMin-numDDChoiceVarsUsed+i]);
		}
		
		// for each i (i = 1 ... max number of nondet. choices)
		for (i = 1; i <= maxChoices; i++) {
			
			// find sections of state space
			// which have exactly i nondet. choices in this module
			equalsi = JDD.Equals(overlaps.copy(), (double)i);
			// if there aren't any for this i, skip the iteration
			if (equalsi.equals(JDD.ZERO)) {
				JDD.Deref(equalsi);
				continue;
			}
			
			// create arrays of size i to store dds
			transDDbits = new JDDNode[i];
			frees = new JDDNode[i];
			for (j = 0; j < i; j++) {
				transDDbits[j] = JDD.Constant(0);
				frees[j] = equalsi.copy();
			}
			
			// go thru each command of the module...
			for (j = 0; j < numCommands; j++) {
				JDDNode guardDD = commandsDDs[j].guard;
				JDDNode upDD = commandsDDs[j].up;
				
				// see if this command's guard overlaps with 'equalsi'
				tmp = JDD.And(guardDD.copy(), equalsi.copy());
				// if it does...
				if (!tmp.equals(JDD.ZERO)) {
					
					// split it up into nondet. choices as necessary
					
					tmp2 = tmp.copy();
					
					// for each possible nondet. choice (1...i) involved...
					for (k = 0; k < i; k ++) {
						// see how much of the command can go in nondet. choice k
						tmp3 = JDD.And(tmp2.copy(), frees[k].copy());
						// if some will fit in...
						if (!tmp3.equals(JDD.ZERO)) {
							frees[k] = JDD.And(frees[k], JDD.Not(tmp3.copy()));
							transDDbits[k] = JDD.Plus(transDDbits[k], JDD.Times(tmp3.copy(), upDD.copy()));
						}
						// take out the bit just put in this choice
						tmp2 = JDD.And(tmp2, JDD.Not(tmp3));
						if (tmp2.equals(JDD.ZERO)) {
							break;
						}
					}
					JDD.Deref(tmp2);
				}
				JDD.Deref(tmp);
			}
			
			// now add the nondet. choices for this value of i
			for (j = 0; j < i; j++) {
				tmp = JDD.SetVectorElement(JDD.Constant(0), ddChoiceVarsUsed, j, 1);
				transDD = JDD.Plus(transDD, JDD.Times(tmp, transDDbits[j]));
				JDD.Deref(frees[j]);
			}
			
			// take the i bits out of 'overlaps'
			overlaps = JDD.Times(overlaps, JDD.Not(equalsi));
		}
		JDD.Deref(overlaps);
		
		// store result
		compDDs.guards = covered;
		compDDs.trans = transDD;
		compDDs.min = synchMin;
		compDDs.max = synchMin + numDDChoiceVarsUsed;
		
		return compDDs;
	}

	/**
	 * Translate the updates part of a command.
	 * <br>[ REFS: <i>result</i>, DEREFS: <i>none</i> ]
	 * @param m the module index
	 * @param l the command index inside the module
	 * @param u the updates AST element
	 * @param synch true if this command is synchronising (named action)
	 * @param guard the guard
	 */
	private UpdateDDs translateUpdates(int m, int l, Updates u, boolean synch, JDDNode guard) throws PrismException
	{
		int i, n;
		Expression p;
		JDDNode dd, udd, pdd = null;
		UpdateDDs updateDDs;
		boolean warned;
		String msg;
		
		// sum up over possible updates
		dd = JDD.Constant(0);
		n = u.getNumUpdates();
		for (i = 0; i < n; i++) {
			// translate a single update
			try {
				updateDDs = translateUpdate(m, u.getUpdate(i), synch, guard);
				udd = updateDDs.up;
			} catch (Exception|StackOverflowError e) {
				JDD.Deref(dd);
				throw e;
			}
			// check for zero update
			warned = false;
			if (udd.equals(JDD.ZERO)) {
				warned = true;
				// Use a PrismLangException to get line numbers displayed
				msg = "Update " + (i+1) + " of command " + (l+1);
				msg += " of module \"" + moduleNames[m] + "\" doesn't do anything";
				mainLog.printWarning(new PrismLangException(msg, u.getUpdate(i)).getMessage());
			}
			// multiply by probability/rate
			p = u.getProbability(i);
			if (p == null) p = Expression.Double(1.0);
			try {
				pdd = translateExpression(p);
			} catch (Exception|StackOverflowError e) {
				JDD.Deref(dd, udd);
				JDD.DerefNonNull(pdd);
				throw e;
			}
			udd = JDD.Times(udd, pdd);
			// check (again) for zero update
			if (!warned && udd.equals(JDD.ZERO)) {
				// Use a PrismLangException to get line numbers displayed
				msg = "Update " + (i+1) + " of command " + (l+1);
				msg += " of module \"" + moduleNames[m] + "\" doesn't do anything";
				mainLog.printWarning(new PrismLangException(msg, u.getUpdate(i)).getMessage());
			}
			dd = JDD.Plus(dd, udd);
		}
		
		return new UpdateDDs(dd);
	}

	/**
	 * Translate an update.
	 * <br>[ REFS: <i>result</i>, DEREFS: <i>none</i> ]
	 * @param m the module index
	 * @param c the update AST element
	 * @param synch true if this command is synchronising (named action)
	 * @param guard the guard
	 */
	private UpdateDDs translateUpdate(int m, Update c, boolean synch, JDDNode guard) throws PrismException
	{
		int n;
		
		// clear varsUsed flag array to indicate no vars used yet
		for (int i = 0; i < numVars; i++) {
			varsUsed[i] = false;
		}
		// take product of clauses
		JDDNode dd = JDD.Constant(1);
		n = c.getNumElements();
		for (int i = 0; i < n; i++) {
			try {
				UpdateDDs udd = translateUpdateElement(m, c, i, synch, guard);
				dd = JDD.Times(dd, udd.up);
			} catch (Exception|StackOverflowError e) {
				JDD.Deref(dd);
				throw e;
			}
		}
		// if a variable from this module or a global variable
		// does not appear in this update assume it does not change value
		// so multiply by its identity matrix
		for (int i = 0; i < numVars; i++) {
			if ((varList.getModule(i) == m || varList.getModule(i) == -1) && !varsUsed[i]) {
				dd = JDD.Times(dd, varIdentities[i].copy());
			}
		}
		
		return new UpdateDDs(dd);
	}

	/**
	 * Translate a single update element, i.e., (x'=...) in an update.
	 * <br>[ REFS: <i>result</i>, DEREFS: <i>none</i> ]
	 * @param m the module index
	 * @param c the Update AST element
	 * @param i the element index for the update element in c
	 * @param synch true if this command is synchronising (named action)
	 * @param guard the guard for this command
	 */
	private UpdateDDs translateUpdateElement(int m, Update c, int i, boolean synch, JDDNode guard) throws PrismException
	{
		// get variable
		String s = c.getVar(i);
		int v = varList.getIndex(s);
		if (v == -1) {
			throw new PrismLangException("Unknown variable \"" + s + "\" in update", c.getVarIdent(i));
		}
		varsUsed[v] = true;
		// check if the variable to be modified is valid
		// (i.e. belongs to this module or is global)
		if (varList.getModule(v) != -1 && varList.getModule(v) != m) {
			throw new PrismLangException("Cannot modify variable \"" + s + "\" from module \"" + moduleNames[m] + "\"", c.getVarIdent(i));
		}
		// print out a warning if this update is in a command with a synchronising
		// action AND it modifies a global variable
//		if (varList.getModule(v) == -1 && synch) {
//			throw new PrismLangException("Synchronous command cannot modify global variable", c.getVarIdent(i));
//		}
		// get some info on the variable
		int l = varList.getLow(v);
		int h = varList.getHigh(v);
		// create dd
		JDDNode tmp1 = JDD.Constant(0);
		for (int j = l; j <= h; j++) {
			tmp1 = JDD.SetVectorElement(tmp1, varDDColVars[v], j - l, j);
		}
		JDDNode tmp2 = translateExpression(c.getExpression(i));
		tmp2 = JDD.Times(tmp2, guard.copy());
		JDDNode cl = JDD.Apply(JDD.EQUALS, tmp1, tmp2);
		cl = JDD.Times(cl, guard.copy());
		// filter out bits not in range
		cl = JDD.Times(cl, varColRangeDDs[v].copy());
		cl = JDD.Times(cl, range.copy());

		return new UpdateDDs(cl);
	}

	/**
	 * Translate an arbitrary expression.
	 * <br>[ REFS: <i>result</i>, DEREFS: <i>none</i> ]
	 * @param e the expression (AST element)
	 */
	private JDDNode translateExpression(Expression e) throws PrismException
	{
		// pass this work onto the Expression2MTBDD object
		// states of interest = JDD.ONE = true = all possible states
		return expr2mtbdd.checkExpressionDD(e, JDD.ONE.copy());
	}

	private void buildDDGame() throws PrismException
	{
		// Build cubes for each player
		ddPlayerCubes = new JDDNode[numPlayers];
		for (int player = 0; player < numPlayers; player++) {
			ddPlayerCubes[player] = ddPlayerVars[player].copy();
			for (int p = 0; p < numPlayers; p++) {
				if (player != p) {
					ddPlayerCubes[player] = JDD.And(ddPlayerCubes[player], JDD.Not(ddPlayerVars[p].copy()));
				}
			}
		}

		// Get the transition matrix for each individual player
		// and find the set of states controlled by each player
		JDDNode ddPlayerTrans[] = new JDDNode[numPlayers];
		JDDNode ddPlayerStates[] = new JDDNode[numPlayers];
		for (int p = 0; p < numPlayers; p++) {
			ddPlayerTrans[p] = JDD.Restrict(trans.copy(), ddPlayerCubes[p].copy());
			ddPlayerStates[p] = JDD.GreaterThan(ddPlayerTrans[p].copy(), 0);
			ddPlayerStates[p] = JDD.ThereExists(JDD.ThereExists(ddPlayerStates[p], allDDNondetVars), allDDColVars);
		}

		// Get transition matrix for actions assigned to no player
		JDDNode ddPlayerNoneCube = JDD.Constant(1);
		for (int p = 0; p < numPlayers; p++) {
			ddPlayerNoneCube = JDD.And(ddPlayerNoneCube, JDD.Not(ddPlayerVars[p].copy()));
		}
		JDDNode ddPlayerNoneTrans = JDD.Restrict(trans.copy(), ddPlayerNoneCube.copy());
		JDDNode ddPlayerNoneStates = JDD.GreaterThan(ddPlayerNoneTrans.copy(), 0);
		ddPlayerNoneStates = JDD.ThereExists(JDD.ThereExists(ddPlayerNoneStates, allDDNondetVars), allDDColVars);

		// Also extract transition rewards for each player (and unassigned)
		JDDNode ddPlayerTransRewards[][] = new JDDNode[numRewardStructs][numPlayers];
		JDDNode ddPlayerNoneTransRewards[] = new JDDNode[numRewardStructs];
		for (int j = 0; j < numRewardStructs; j++) {
			ddPlayerTransRewards[j] = new JDDNode[numPlayers];
			for (int p = 0; p < numPlayers; p++) {
				ddPlayerTransRewards[j][p] = JDD.Restrict(transRewards[j].copy(), ddPlayerCubes[p].copy());
			}
			ddPlayerNoneTransRewards[j] = JDD.Restrict(transRewards[j].copy(), ddPlayerNoneCube.copy());
		}

		// Check for overlaps in player-controlled states
		JDDNode ddPlayerOverlaps = JDD.Constant(0);
		for (int p = 0; p < numPlayers; p++) {
			ddPlayerOverlaps = JDD.Plus(ddPlayerOverlaps, ddPlayerStates[p].copy());
		}
		ddPlayerOverlaps = JDD.GreaterThan(ddPlayerOverlaps, 1);
//		if (!ddPlayerOverlaps.equals(JDD.ZERO)) {
//			JDD.Deref(ddPlayerOverlaps);
//			JDD.PrintVector(ddPlayerOverlaps, allDDRowVars, JDD.LIST);
//			throw new PrismException("There are states with multiple players enabled");
//		}
		JDD.Deref(ddPlayerOverlaps);

		// Add transitions from unassigned actions to the relevant player
		// i.e., to the player that owns the state in which they occur.
		// Do this for both the transition matrix and any transition rewards
		// (TODO: check that there is no nondeterminism, left afterwards)
		for (int p = 0; p < numPlayers; p++) {
			ddPlayerTrans[p] = JDD.Plus(ddPlayerTrans[p], JDD.Times(ddPlayerNoneTrans.copy(), ddPlayerStates[p].copy()));
			ddPlayerNoneTrans = JDD.Times(ddPlayerNoneTrans, JDD.Not(ddPlayerStates[p].copy()));
			for (int j = 0; j < numRewardStructs; j++) {
				ddPlayerTransRewards[j][p] = JDD.Plus(ddPlayerTransRewards[j][p], JDD.Times(ddPlayerNoneTransRewards[j].copy(), ddPlayerStates[p].copy()));
				ddPlayerNoneTransRewards[j] = JDD.Times(ddPlayerNoneTransRewards[j], JDD.Not(ddPlayerStates[p].copy()));
			}
		}
		// Add any remaining unassigned actions to player 1
		ddPlayerTrans[0] = JDD.Plus(ddPlayerTrans[0], ddPlayerNoneTrans.copy());
		for (int j = 0; j < numRewardStructs; j++) {
			ddPlayerTransRewards[j][0] = JDD.Plus(ddPlayerTransRewards[j][0], ddPlayerNoneTransRewards[j].copy());
		}
		JDD.Deref(ddPlayerStates[0]);
		ddPlayerStates[0] = JDD.GreaterThan(ddPlayerTrans[0].copy(), 0);
		ddPlayerStates[0] = JDD.ThereExists(JDD.ThereExists(ddPlayerStates[0], allDDNondetVars), allDDColVars);

		// Reconstruct trans
		JDD.Deref(trans);
		trans = JDD.Constant(0);
		for (int p = 0; p < numPlayers; p++) {
			trans = JDD.Plus(trans, JDD.Times(ddPlayerCubes[p].copy(), ddPlayerTrans[p].copy()));
		}

		// Reconstruct trans rewards
		for (int j = 0; j < numRewardStructs; j++) {
			JDD.Deref(transRewards[j]);
			transRewards[j] = JDD.Constant(0);
			for (int p = 0; p < numPlayers; p++) {
				transRewards[j] = JDD.Plus(transRewards[j], JDD.Times(ddPlayerCubes[p].copy(), ddPlayerTransRewards[j][p].copy()));
			}
		}

		// Derefs
		JDD.DerefArray(ddPlayerTrans, numPlayers);
		JDD.DerefArray(ddPlayerStates, numPlayers);
		JDD.Deref(ddPlayerNoneCube);
		JDD.Deref(ddPlayerNoneTrans);
		JDD.Deref(ddPlayerNoneStates);
		for (int j = 0; j < numRewardStructs; j++) {
			JDD.DerefArray(ddPlayerTransRewards[j], numPlayers);
			JDD.Deref(ddPlayerNoneTransRewards[j]);
		}
	}

	// build state and transition rewards
	
	private void computeRewards(SystemDDs sysDDs) throws PrismException
	{
		RewardStruct rs;
		int i, j, k, n;
		double d;
		String synch, s;
		JDDNode rewards, states, item;
		ComponentDDs compDDs;
		
		// how many reward structures?
		numRewardStructs = modulesFile.getNumRewardStructs();
		
		// initially rewards zero
		stateRewards = new JDDNode[numRewardStructs];
		transRewards = new JDDNode[numRewardStructs];
		for (j = 0; j < numRewardStructs; j++) {
			stateRewards[j] = JDD.Constant(0);
			sysDDs.ind.rewards[j] = JDD.Constant(0);
			for (i = 0; i < numSynchs; i++) sysDDs.synchs[i].rewards[j] = JDD.Constant(0);
		}
		
		// for each reward structure...
		for (j = 0; j < numRewardStructs; j++) {
			
			// get reward struct
			rs = modulesFile.getRewardStruct(j);
			
			// work through list of items in reward struct
			n = rs.getNumItems();
			for (i = 0; i < n; i++) {
				
				// translate states predicate and reward expression
				states = translateExpression(rs.getStates(i));
				rewards = translateExpression(rs.getReward(i));
				
				// first case: item corresponds to state rewards
				synch = rs.getSynch(i);
				if (synch == null) {
					// restrict rewards to relevant states
					item = JDD.Times(states, rewards);
					// check for infinite/NaN/negative rewards
					double dmin = JDD.FindMin(item);
					double dmax = JDD.FindMax(item);
					if (!Double.isFinite(dmin)) {
						s = "Reward structure item contains non-finite rewards (" + dmin + ").";
						s += "\nNote that these may correspond to states which are unreachable.";
						s += "\nIf this is the case, try strengthening the predicate.";
						throw new PrismLangException(s, rs.getRewardStructItem(i));
					}
					if (!Double.isFinite(dmax)) {
						s = "Reward structure item contains non-finite rewards (" + dmax + ").";
						s += "\nNote that these may correspond to states which are unreachable.";
						s += "\nIf this is the case, try strengthening the predicate.";
						throw new PrismLangException(s, rs.getRewardStructItem(i));
					}
					/*if (dmin < 0) {
						s = "Reward structure item contains negative rewards (" + dmin + ").";
						s += "\nNote that these may correspond to states which are unreachable.";
						s += "\nIf this is the case, try strengthening the predicate.";
						throw new PrismLangException(s, rs.getRewardStructItem(i));
					}*/
					// add to state rewards
					stateRewards[j] = JDD.Plus(stateRewards[j], item);
				}
				
				// second case: item corresponds to transition rewards
				else {
					// work out which (if any) action this is for
					if ("".equals(synch)) {
						compDDs = sysDDs.ind;
					} else if ((k = synchs.indexOf(synch)) != -1) {
						compDDs = sysDDs.synchs[k];
					} else {
						throw new PrismLangException("Invalid action name \"" + synch + "\" in reward structure item", rs.getRewardStructItem(i));
					}
					// identify corresponding transitions
					// (for dtmcs/ctmcs, keep actual values - need to weight rewards; for mdps just store 0/1)
					if (modelType == ModelType.MDP || modelType == ModelType.SMG) {
						item = JDD.GreaterThan(compDDs.trans.copy(), 0);
					} else {
						item = compDDs.trans.copy();
					}
					// restrict to relevant states
					item = JDD.Times(item, states);
					// multiply by reward values
					item = JDD.Times(item, rewards);
					// check for infinite/NaN/negative rewards
					double dmin = JDD.FindMin(item);
					double dmax = JDD.FindMax(item);
					if (!Double.isFinite(dmin)) {
						s = "Reward structure item contains non-finite rewards (" + dmin + ").";
						s += "\nNote that these may correspond to states which are unreachable.";
						s += "\nIf this is the case, try strengthening the predicate.";
						throw new PrismLangException(s, rs.getRewardStructItem(i));
					}
					if (!Double.isFinite(dmax)) {
						s = "Reward structure item contains non-finite rewards (" + dmax + ").";
						s += "\nNote that these may correspond to states which are unreachable.";
						s += "\nIf this is the case, try strengthening the predicate.";
						throw new PrismLangException(s, rs.getRewardStructItem(i));
					}
					/*if (dmin < 0) {
						s = "Reward structure item contains negative rewards (" + dmin + ").";
						s += "\nNote that these may correspond to states which are unreachable.";
						s += "\nIf this is the case, try strengthening the predicate.";
						throw new PrismLangException(s, rs.getRewardStructItem(i));
					}*/
					// add result to rewards
					compDDs.rewards[j] = JDD.Plus(compDDs.rewards[j], item);
				}
			}
		}
	}
	
	// calculate dd for initial state(s)
	
	private void buildInitialStates() throws PrismException
	{
		int i;
		JDDNode tmp;
		
		// first, handle case where multiple initial states specified with init...endinit
		if (modulesFile.getInitialStates() != null) {
			start = translateExpression(modulesFile.getInitialStates());
			JDD.Ref(range);
			start = JDD.And(start, range);
			if (start.equals(JDD.ZERO)) throw new PrismLangException("No initial states: \"init\" construct evaluates to false", modulesFile.getInitialStates());
		}
		// second, handle case where initial state determined by init values for variables
		else {
			start = JDD.Constant(1);
			for (i = 0; i < numVars; i++) {
				Object startObj = modulesFile.getVarDeclaration(i).getStartOrDefault().evaluate(constantValues);
				try {
					int startInt = varList.encodeToInt(i, startObj);
					tmp = JDD.SetVectorElement(JDD.Constant(0), varDDRowVars[i], startInt, 1);
					start = JDD.And(start, tmp);
				} catch (PrismLangException e) {
					// attach initial value spec for better error reporting
					e.setASTElement(modulesFile.getVarDeclaration(i).getStart());
					throw e;
				}
			}
		}
	}
	
	// symmetrification
	
	private void doSymmetry(ModelSymbolic model) throws PrismException
	{
		JDDNode tmp, transNew, reach, trans, transRewards[];
		int i, j, k, numSwaps;
		boolean done;
		long clock;
		String ss[];
		
		// parse symmetry reduction parameters
		ss = prism.getSettings().getString(PrismSettings.PRISM_SYMM_RED_PARAMS).split(" ");
		if (ss.length != 2) throw new PrismException ("Invalid parameters for symmetry reduction");
		try {
			numModulesBeforeSymm = Integer.parseInt(ss[0].trim());
			numModulesAfterSymm = Integer.parseInt(ss[1].trim());
		}
		catch (NumberFormatException e) {
			throw new PrismException("Invalid parameters for symmetry reduction");
		}

		clock = System.currentTimeMillis();
		
		// get a copies of model (MT)BDDs
		reach = model.getReach();
		JDD.Ref(reach);
		trans =  model.getTrans();
		JDD.Ref(trans);
		transRewards = new JDDNode[numRewardStructs];
		for (i = 0; i < numRewardStructs; i++) {
			transRewards[i] =  model.getTransRewards(i);
			JDD.Ref(transRewards[i]);
		}
		
		mainLog.print("\nApplying symmetry reduction...\n");
		
		//identifySymmetricModules();
		numSymmModules = numModules - (numModulesBeforeSymm + numModulesAfterSymm);
		computeSymmetryFilters(reach);
		
		// compute number of local states
// 		JDD.Ref(reach);
// 		tmp = reach;
// 		for (i = 0; i < numModules; i++) {
// 			if (i != numModulesBeforeSymm) tmp = JDD.ThereExists(tmp, moduleDDRowVars[i]);
// 		}
// 		tmp = JDD.ThereExists(tmp, globalDDRowVars);
// 		mainLog.println("Local states: " + (int)JDD.GetNumMinterms(tmp, moduleDDRowVars[numModulesBeforeSymm].n()));
// 		JDD.Deref(tmp);
		
		//ODDNode odd = ODDUtils.BuildODD(reach, allDDRowVars);
		//try {sparse.PrismSparse.NondetExport(trans, allDDRowVars, allDDColVars, allDDNondetVars, odd, Prism.EXPORT_PLAIN, "trans-full.tra"); } catch (FileNotFoundException e) {}
		
		mainLog.println("\nNumber of states before before symmetry reduction: " + model.getNumStatesString());
		mainLog.println("DD sizes before symmetry reduction:");
		
		// trans - rows
		mainLog.print("trans: ");
		mainLog.println(JDD.GetInfoString(trans, (modelType==ModelType.MDP)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
		JDD.Ref(symm);
		trans = JDD.Apply(JDD.TIMES, trans, symm);
		//mainLog.print("trans (symm): ");
		//mainLog.println(JDD.GetInfoString(trans, (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
		
		// trans rewards - rows
		for (k = 0; k < numRewardStructs; k++) {
			mainLog.print("transrew["+k+"]: ");
			mainLog.println(JDD.GetInfoString(transRewards[k], (modelType==ModelType.MDP)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
			JDD.Ref(symm);
			transRewards[k] = JDD.Apply(JDD.TIMES, transRewards[k], symm);
			//mainLog.print("transrew["+k+"] (symm): ");
			//mainLog.println(JDD.GetInfoString(transRewards[k], (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
		}
		
		mainLog.println("Starting quicksort...");
		done = false;
		numSwaps = 0;
		for (i = numSymmModules; i > 1 && !done; i--) {
			// store trans from previous iter
			JDD.Ref(trans);
			transNew = trans;
			for (j = 0; j < i-1; j++) {
				
				// are there any states where j+1>j+2?
				if (nonSymms[j].equals(JDD.ZERO)) continue;
				
				// identify offending block in trans
				JDD.Ref(transNew);
				JDD.Ref(nonSymms[j]);
				tmp = JDD.Apply(JDD.TIMES, transNew, JDD.PermuteVariables(nonSymms[j], allDDRowVars, allDDColVars));
				//mainLog.print("bad block: ");
				//mainLog.println(JDD.GetInfoString(tmp, (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
				
				if (tmp.equals(JDD.ZERO)) { JDD.Deref(tmp); continue; }
				numSwaps++;
				mainLog.println("Iteration "+(numSymmModules-i+1)+"."+(j+1));
				
				// swap
				tmp = JDD.SwapVariables(tmp, moduleDDColVars[numModulesBeforeSymm+j], moduleDDColVars[numModulesBeforeSymm+j+1]);
				//mainLog.print("bad block (swapped): ");
				//mainLog.println(JDD.GetInfoString(tmp, (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
				
				// insert swapped block
				JDD.Ref(nonSymms[j]);
				JDD.Ref(tmp);
				transNew = JDD.ITE(JDD.PermuteVariables(nonSymms[j], allDDRowVars, allDDColVars), JDD.Constant(0), JDD.Apply(JDD.PLUS, transNew, tmp));
				//mainLog.print("trans (symm): ");
				//mainLog.println(JDD.GetInfoString(transNew, (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
				JDD.Deref(tmp);
				
				for (k = 0; k < numRewardStructs; k++) {
					// identify offending block in trans rewards
					JDD.Ref(transRewards[k]);
					JDD.Ref(nonSymms[j]);
					tmp = JDD.Apply(JDD.TIMES, transRewards[k], JDD.PermuteVariables(nonSymms[j], allDDRowVars, allDDColVars));
					//mainLog.print("bad block: ");
					//mainLog.println(JDD.GetInfoString(tmp, (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
					
					// swap
					tmp = JDD.SwapVariables(tmp, moduleDDColVars[numModulesBeforeSymm+j], moduleDDColVars[numModulesBeforeSymm+j+1]);
					//mainLog.print("bad block (swapped): ");
					//mainLog.println(JDD.GetInfoString(tmp, (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
					
					// insert swapped block
					JDD.Ref(nonSymms[j]);
					JDD.Ref(tmp);
					transRewards[k] = JDD.ITE(JDD.PermuteVariables(nonSymms[j], allDDRowVars, allDDColVars), JDD.Constant(0), JDD.Apply(JDD.PLUS, transRewards[k], tmp));
					//mainLog.print("transrew["+k+"] (symm): ");
					//mainLog.println(JDD.GetInfoString(transRewards[k], (type==ModulesFile.NONDETERMINISTIC)?(allDDRowVars.n()*2+allDDNondetVars.n()):(allDDRowVars.n()*2)));
					JDD.Deref(tmp);
				}
			}
			
			if (transNew.equals(trans)) {
				done = true;
			}
			JDD.Deref(trans);
			trans = transNew;
		}
		
		// reset (MT)BDDs in model
		model.resetTrans(trans);
		for (i = 0; i < numRewardStructs; i++) {
			model.resetTransRewards(i, transRewards[i]);
		}
		
		// reset reach bdd, etc.
		JDD.Ref(symm);
		reach = JDD.And(reach, symm);
		
		model.setReach(reach);
		model.filterReachableStates();
		
		clock = System.currentTimeMillis() - clock;
		mainLog.println("Symmetry complete: " + (numSymmModules-i) + " iterations, " + numSwaps + " swaps, " + clock/1000.0 + " seconds");
	}

	private void computeSymmetryFilters(JDDNode reach) throws PrismException
	{
		int i;
		JDDNode tmp;
		
		// array for non-symmetric parts
		nonSymms = new JDDNode[numSymmModules-1];
		// dd for all symmetric states
		JDD.Ref(reach);
		symm = reach;
		// loop through symmetric module pairs
		for (i = 0; i < numSymmModules-1; i++) {
			// (locally) symmetric states, i.e. where i+1 <= i+2
			tmp = JDD.VariablesLessThanEquals(moduleDDRowVars[numModulesBeforeSymm+i], moduleDDRowVars[numModulesBeforeSymm+i+1]);
			// non-(locally)-symmetric states
			JDD.Ref(tmp);
			JDD.Ref(reach);
			nonSymms[i] = JDD.And(JDD.Not(tmp), reach);
			// all symmetric states
			symm = JDD.And(symm, tmp);
		}
	}

	// old version of computeSymmetryFilters()
	/*private void computeSymmetryFilters() throws PrismException
	{
		int i, j, k, n;
		String varNames[][] = null;
		JDDNode tmp;
		Expression expr, exprTmp;
		
		// get var names for each symm module
		n = modulesFile.getModule(numModulesBeforeSymm).getNumDeclarations();
		varNames = new String[numModules][];
		for (i = numModulesBeforeSymm; i < numModulesBeforeSymm+numSymmModules; i++) {
			varNames[i-numModulesBeforeSymm] = new String[n];
			j = 0;
			while (j < numVars && varList.getModule(j) != i) j++;
			for (k = 0; k < n; k++) {
				varNames[i-numModulesBeforeSymm][k] = varList.getName(j+k);
			}
		}
		
		// array for non-symmetric parts
		nonSymms = new JDDNode[numSymmModules-1];
		// dd for all symmetric states
		JDD.Ref(reach);
		symm = reach;
		// loop through symmetric module pairs
		for (i = 0; i < numSymmModules-1; i++) {
			// expression for (locally) symmetric states, i.e. where i+1 <= i+2
			expr = new ExpressionTrue();
			for (j = varNames[0].length-1; j >= 0 ; j--) {
				exprTmp = new ExpressionAnd();
				((ExpressionAnd)exprTmp).addOperand(new ExpressionBrackets(new ExpressionRelOp(new ExpressionVar(varNames[i][j], 0), "=", new ExpressionVar(varNames[i+1][j], 0))));
				((ExpressionAnd)exprTmp).addOperand(new ExpressionBrackets(expr));
				expr = exprTmp;
				exprTmp = new ExpressionOr();
				((ExpressionOr)exprTmp).addOperand(new ExpressionBrackets(new ExpressionRelOp(new ExpressionVar(varNames[i][j], 0), "<", new ExpressionVar(varNames[i+1][j], 0))));
				((ExpressionOr)exprTmp).addOperand(expr);
				expr = exprTmp;
			}
			mainLog.println(expr);
			// bdd for (locally) symmetric states, i.e. where i+1 <= i+2
			tmp = expr2mtbdd.translateExpression(expr);
			// non-(locally)-symmetric states
			JDD.Ref(tmp);
			JDD.Ref(reach);
			nonSymms[i] = JDD.And(JDD.Not(tmp), reach);
			// all symmetric states
			symm = JDD.And(symm, tmp);
		}
	}*/
}

//------------------------------------------------------------------------------
